#!/bin/bash

log_msg() {
  # 使用 logger 命令将日志写入 syslog
  logger -t hdmi-hotplug "$1"
}

# 检查 Xorg 是否运行，如果没有则启动
if ! pidof Xorg > /dev/null; then
  log_msg "Xorg 没有运行，等待 Xorg 启动..."
  while ! pidof Xorg > /dev/null; do
    sleep 0.1
  done
  log_msg "Xorg 启动完成"
fi

set -euf ; (set -o pipefail)2>/dev/null && set -o pipefail

vprefs=/etc/X11/vmode.prefs

get_xauth() {
  while [ $# -gt 0 ]
  do
    case "$1" in
    -auth)
      echo "$2" ; return 0
      ;;
    esac
    shift
  done
  return 1
}

get_display() {
  if [ $# -gt 1 ] ; then
    case "$2" in
      :*) echo "$2" ; return
    esac
  fi
  # Default to display :0
  echo ":0"
}

if [ -z "${DISPLAY:-}" ] ; then
  if ! xpid=$(pidof Xorg) ; then
    log_msg "Xorg not running"
    exit
  fi
  xorg_args="$(tr '\0' ' ' </proc/$xpid/cmdline)"
  DISPLAY=$(get_display $xorg_args)
  if ! XAUTHORITY=$(get_xauth $xorg_args) ; then
    log_msg "Unable to initialize XAUTHORITY"
    exit
  fi
  export DISPLAY XAUTHORITY
fi

xrandr_output=$(xrandr) || exit 1
log_msg "xrandr 输出: $xrandr_output"

# 获取所有连接的显示器
outputs=$(echo "$xrandr_output" | awk '$2 == "connected" { print $1 }')
if [ -z "$outputs" ] ; then
  log_msg "No connected outputs"
  exit
fi

# 选择主显示器（默认选择第一个连接的设备）
primary_output=$(echo "$outputs" | head -n1)
log_msg "主显示器: $primary_output"

# 获取主显示器的分辨率
primary_resolution=$(echo "$xrandr_output" | grep -A1 "$primary_output" | tail -n1 | awk '{print $1}')
log_msg "主显示器分辨率: $primary_resolution"

# 让其他所有显示器镜像主显示器，并设置分辨率
for output in $outputs
do
  if [ "$output" != "$primary_output" ]; then
    xrandr --output $output --same-as $primary_output --mode $primary_resolution
    log_msg "设置 $output 复制镜像 $primary_output，分辨率 $primary_resolution"
  fi
done

sleep 1
xrefresh
