#! /bin/bash
set -eu
ROOT_DIR=$(dirname $(readlink -f $0))

log_debug() {
    [[ -L /dev/log || -f /dev/log ]] && logger -p local3.debug -t build_system[$$] $@
    echo "[DEBUG] $@"
}
log_info() {
    [[ -L /dev/log || -f /dev/log ]] && logger -p local3.info -t build_system[$$] $@
    echo "[INFO] $@"
}
log_warning() {
    [[ -L /dev/log || -f /dev/log ]] && logger -p local3.warning -t build_system[$$] $@
    echo "[WARN] $@"
}
log_error() {
    [[ -L /dev/log || -f /dev/log ]] && logger -p local3.err -t build_system[$$] $@
    echo "[ERROR] $@" >&2
}

# 判断系统版本是否符合
system_version=$(lsb_release -rs)
if [ "$(echo "$system_version != 22.04" | bc)" -ne 0 ]; then
    log_info "The system version does not match the environment package version. Please run the script in the xubuntu 22.04 environment."
    exit 1
fi

if grep 'local3.* /var/log/arcs_system.log' /etc/rsyslog.conf &>/dev/null;then
    log_info "The rsyslog config already exists"
else
    log_info "Add rsyslog config"
    echo -e "\nlocal3.* /var/log/arcs_system.log" >> /etc/rsyslog.conf
    systemctl restart rsyslog ||:
fi


source /etc/default/arcs ||:

[ -f ${ROOT_DIR}/version ] && source ${ROOT_DIR}/version

VERSION=${VERSION:-0.0.0}

generate_build_system_version_file() {
    log_info "Generate config file '/etc/default/setup_env '"
    echo "VERSION=${VERSION}" > /etc/default/setup_env
}

generate_build_system_version_file

log_info start installing version ${VERSION} in arcs environment...

if ! [ $(id -u) = 0 ]; then
    log_info "$0: The script need to be run as root."
    exit 1
fi

# 判断是否为虚拟机环境
if lscpu | grep -E "VMware|KVM" > /dev/null; then
    machine_type="virtual"
    log_info "Running on virtual machine(VMware|KVM)"
elif [ "${CONTAINER:-unbound}" != "unbound" ];then
    machine_type="container"
    log_info "Running on container"
else
    log_info "Running on physical machine"
    machine_type="physical"
fi

# 0安装
# 0.1触摸屏驱动(注意要等待执行结束,最后生成触摸屏配置数据库文件)
install_pmLinux(){
    log_info "install_pmLinux.."
    cd ${ROOT_DIR}
    # tar -xvf pmLinux-Ubuntu_14.04_24_04_v4_7_7_R15_x86_64_aarch64_20240820_touch.tar
    cd ./pmLinux-Ubuntu
    log_info $(./install.sh)
    cd ${ROOT_DIR}
    # rm -rf pmLinux-Ubuntu
}

# 0.2 安装rtl8812au无线驱动
install_rtl8812au(){
    log_info "install_rtl8812au..."
    depmod -a || true
    modprobe 8821au
}

# 0.5 拷贝libstdc++.so.6,libxkbcommon库文件
copy_files() {
    log_info "copy_files.."
    chown -R root:root chroot/
    chmod +x /usr/bin/*
    rsync -a -vv --exclude 'lib/*' --exclude 'sbin/*' --exclude 'lib/' --exclude 'sbin/' chroot/ /
    rsync -a -vv chroot/lib/ /lib/
    rsync -a -vv chroot/sbin/ /sbin/
    rsync -a -vv chroot/usr/local/nginx/sbin/ /usr/local/nginx/sbin/
    chmod +x /usr/bin/*
}

# 保持主机名一致
fix_hosts() {
    log_info "fix_hosts.."
    new_hostname=$(cat /etc/hostname)
    new_ip="127.0.1.1"
    sed -i "2s/.*/$new_ip\t$new_hostname/" /etc/hosts
}

clean_srs_and_aubope() {
    rm -f /etc/rc.local
    test -d /root/auborobotstudio && rm -rf /root/auborobotstudio
    test -f /root/aubostart && rm -f /usr/bin/aubostart
    sed -i '/aubostart/d' /etc/profile
}

## NOTE: 必须联网安装 xubuntu-desktop
install_xubuntu_desktop() {
    log_info "Configure offline installation source"
    # 配置离线安装源
    echo "deb file:///${ROOT_DIR}/offline-packages archives/"| tee /etc/apt/sources.list
    
    mv /var/lib/dpkg/info/ /var/lib/dpkg/info_old/
    mkdir /var/lib/dpkg/info/
    apt-get update
    apt install xubuntu-desktop -y
    mv /var/lib/dpkg/info/* /var/lib/dpkg/info_old/
    rm -rf /var/lib/dpkg/info
    mv /var/lib/dpkg/info_old/ /var/lib/dpkg/info/
}

cpu_iso() {
    log_info "cpu_iso.."
    # 1实时性配置
    # 1.1 孤立cpu1
    sed -i 's/^GRUB_CMDLINE_LINUX_DEFAULT=.*$/GRUB_CMDLINE_LINUX_DEFAULT="quiet splash kthread_cpus=0,2,3 irqaffinity=0,2,3 isolcpus=1"/' /etc/default/grub
    update-grub
}

enable_pulseAudio() {
    log_info "enable_pulseAudio.."
    mkdir -p /root/.config/autostart
    if [ ! -f /root/.config/autostart/pulseaudio.desktop ];then
    cat > /root/.config/autostart/pulseaudio.desktop <<EOF
[Desktop Entry]
Type=Application
Exec=pulseaudio --start
Name=PulseAudio
Comment=PulseAudio Sound Server
X-GNOME-Autostart-enabled=true
EOF
    fi
}

cpu_affinity() {
    log_info "cpu_affinity.."
    systemctl stop irqbalance.service

    no_rt_kernerl=3
    # 1.2 更改中断的CPU亲和(终端不显示错误信息,因为有些终端不允许移动到别的核)
    for file in `find /proc/irq -name "smp_affinity_list"`; do
        echo ${no_rt_kernerl} > ${file} > /dev/null;
    done
    
    # `KWorker`的CPU亲和(Linux kernel中将工作推后执行的一种机制)
    for file in `find /sys/devices/virtual/workqueue -name "cpumask"`; do
        echo ${no_rt_kernerl} > ${file} 2>/dev/null;
    done
    echo ${no_rt_kernerl} > /sys/devices/virtual/workqueue/cpumask
}

cpu_freq() {
    log_info "cpu_afcpu_freqfinity.."
    sudo dpkg -i cpu_freq/*.deb
    # 1.3 禁用cpu调频
    
    # 将cpu设置为性能模式
    np=`cat /proc/cpuinfo | grep processor | wc -l`
    let num=($np -1)
    for i in $(seq 0 $num)
    do
        cpufreq-set -c $i -g performance
    done
    cpufreq-info
}

# sysfs.conf documentation link https://manpages.debian.org/testing/sysfsutils/sysfs.conf.5.en.html
config_sysfsutils() {
    log_info "add parameters to sysfsutils"
    sudo dpkg -i sysfsutils/*.deb
    cp -av sysfsutils/sysfs.conf /etc/sysfs.conf
    # Restart sysfsutils service
    echo "Applying configuration changes..."
    if systemctl restart sysfsutils.service 2>/dev/null; then
        echo "sysfsutils service restarted successfully"
    elif /etc/init.d/sysfsutils restart 2>/dev/null; then
        echo "sysfsutils service restarted"
    else
        echo "Warning: Failed to restart sysfsutils service, please restart manually to apply changes"
    fi

    echo "Configuration complete! All CPU cores set to performance mode"
}

disable_culling() {
    log_info "disable_culling.."
    # 2.解决鼠标图标重影问题
    echo "CLUTTER_PAINT=disable-clipped-redraws:disable-culling" >> /etc/environment
}

sshd_config() {
    log_info "sshd_config.."
    # 3.配置默认ssh端口 为8822, 允许 root 远程登录
    sed -i 's/#Port 22/Port 8822/;s/#PermitRootLogin prohibit-password/PermitRootLogin yes/' /etc/ssh/sshd_config
}

root_auto_login() {
    log_info "root_auto_login.."
    # 4.修改用户密码
    echo 'root:bestcobot' | chpasswd
    
    # 5.配置 root 自动登录
    test -f /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf && rm /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    echo "[Seat:*]" >> /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    echo "user-session=xubuntu" >> /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    echo "autologin-guest=false" >> /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    echo "autologin-user=root" >> /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    echo "autologin-user-timeout=0" >> /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    echo "greeter-session=lightdm-gtk-greeter" >> /usr/share/lightdm/lightdm.conf.d/60-xubuntu.conf
    
    test -f /etc/lightdm/lightdm.conf && rm /etc/lightdm/lightdm.conf
    echo "[Seat:*]" >> /etc/lightdm/lightdm.conf
    echo "autologin-session=xubuntu" >> /etc/lightdm/lightdm.conf
    echo "autologin-guest=false" >> /etc/lightdm/lightdm.conf
    echo "autologin-user=root" >> /etc/lightdm/lightdm.conf
    echo "autologin-user-timeout=0" >> /etc/lightdm/lightdm.conf
    
    # 修改/root/.profile文件
    sed -i 's/mesg n/tty -s \&\& mesg n/' /root/.profile
}

xinput_map() {
    log_info "xinput_map.."
    # 6.开机自启动映射 xinput (显示器和触摸屏先建立映射关系)
    mkdir -p /root/.config/autostart
cat <<EOF > /root/.config/autostart/xinput.desktop
[Desktop Entry]
Type=Application
Exec=xfce4-terminal -e bash -c 'xinput map-to-output "DIALOGUE INC PenMount USB" eDP-1'
Hidden=false
NoDisplay=false
X-GNOME-Autostart-enabled=true
EOF
}

disable_update_and_apport() {
    log_info "disable_update_and_apport.."
    # 8.关闭系统崩溃提示,关闭系统自动更新升级，删除更新通知程序
    sed -i "s/enabled=1/enabled=0/g" /etc/default/apport
    sed -i "s/1/0/g" /etc/apt/apt.conf.d/10periodic
    sed -i "s/1/0/g" /etc/apt/apt.conf.d/20auto-upgrades
    sudo apt-get remove update-notifier -y
}

optimize_networking_service(){
    log_info "optimize_networking_service.."
    sed -i "s/TimeoutStartSec=5min/TimeoutStartSec=5s/g" /lib/systemd/system/NetworkManager.service
    
    # 禁用系统启动时候等待网络连接的服务
    systemctl disable NetworkManager-wait-online.service
}

disable_bluetooth() {
    log_info "disable_bluetooth brltty.."
    # 禁用蓝牙服务
    systemctl disable --now bluetooth.service
    systemctl mask bluetooth.service
    # 禁用盲文服务
    systemctl disable --now brltty.service
    systemctl disable --now brltty-udev.service
    systemctl mask brltty.service
    systemctl mask brltty-udev.service
}

disable_useless_service() {
    log_info "disable useless service.."
    for i in {1..3}; do
        for p in $(snap list | awk 'NR>1{print $1}'); do
        sudo snap remove $p 2>/dev/null || true
        done
    done
    service_list="
        snapd.service 
        dbus-org.bluez.service 
        fwupd.service 
        avahi-daemon.socket 
        avahi-daemon.service 
        apport.service
        whoopsie.target 
        cups.service 
        snapd.seeded.service
        cpufrequtils.service
        ModemManager.service
        systemd-networkd 
        networkd-dispatcher.service
        plymouth-quit-wait.service
        accounts-daemon.service
        irqbalance.service"
    for service_name in $(echo ${service_list});do
        systemctl disable --now $service_name || true
    done
    
    mkdir -p /etc/xdg/autostart_backup
    rsync -av /etc/xdg/autostart/ /etc/xdg/autostart_backup/
    find /etc/xdg/autostart -maxdepth 1 -type f \( ! -name 'onboard-autostart.desktop' ! -name 'pulseaudio.desktop' \) -exec rm -v {} +
}

config_vnc() {
    log_info "config_vnc.."
    sudo dpkg -i x11vnc/*.deb

    mypasswd="bestcobot"
    x11vnc -storepasswd $mypasswd /root/.vnc/passwd
    chown -R root /root/.vnc
    chmod 0600 /root/.vnc/passwd

    sudo systemctl daemon-reload
    sudo systemctl enable x11vnc.service
    sudo systemctl start x11vnc.service
}

set_tcp_fin_timeout(){
    log_info "set_tcp_fin_timeout.."
    # 设置端口释放后的等待时间为30s
    sysctl -w net.ipv4.tcp_fin_timeout=30
    # 释放TIME_WAIT端口给新连接使用
    sysctl -w net.ipv4.tcp_timestamps=1
}

# u盘自动挂载识别并执行程序
config_autousb() {
    log_info "config_autousb..."
    chmod a+x /usr/local/bin/usbplug.sh
    systemctl daemon-reload
    systemctl restart udev
}

# 添加系统实时监控
config_monitor() {
    log_info "config_monitor..."
    chmod a+x /usr/local/bin/record_system_info.sh
    systemctl daemon-reload
    systemctl enable system_info_monitor
}

# 添加系统跑分
config_sysbench() {
    log_info "config_sysbench..."
    chmod a+x /usr/local/bin/system-benchmark.sh
}

# 添加exfat
config_exfat() {   
    log_info "config_exfat.."
    sudo dpkg -i exfat/*.deb
}

# 添加7z
config_7zip() {
    log_info "config_7zip.."
    sudo chmod a+x 7z/7zz
    sudo cp -av 7z/7zz /usr/bin/7z
    sudo cp -av 7z/7zz /usr/bin/7zz
}

# 限制系统日志大小
config_logrotate() {
    log_info "config_logrotate"
    chmod 644 /etc/logrotate.d/rsyslog
    chmod 644 /etc/logrotate.d/monitor
    sed -i '4,6 s/^/#/' /etc/cron.daily/logrotate
    if [[ $(grep logrotate /etc/crontab|wc -l) == "0" ]];then
        echo "*/2 * * * *   root    /bin/bash /etc/cron.daily/logrotate > /dev/null 2>&1" >> /etc/crontab 
    fi
}

# 重置connection
config_network() {
    log_info "change the network card method to link-local"
    # 删除所有网络连接
    while IFS= read -r connection_name; do
        nmcli connection delete "$connection_name" >/dev/null 2>&1
    done < <(nmcli -t -f NAME connection show)
    # 配置有线网卡自动连接
    for ethname in $(ifconfig | grep -oP '^\w+' | grep -v 'lo' | grep -v "wl");do 
        nmcli connection add con-name "Wired Connection ${ethname}" type ethernet autoconnect yes ifname ${ethname}
    done
}

# 配置hdmi热重载
config_hdmihotreload() {
    log_info "config_hdmihotreload"
    chmod a+x ${ROOT_DIR}/hdmi-hotreload/hdmi-hotplug.sh
    cp -av ${ROOT_DIR}/hdmi-hotreload/hdmi-hotplug.sh /usr/local/bin/
    cp -av ${ROOT_DIR}/hdmi-hotreload/99-hdmi-hotplug.rules /etc/udev/rules.d/
}

# 配置开机执行脚本
config_rclocal() {
    log_info "enable rc.local"
    chmod a+x /etc/rc.local
    chmod a+x /etc/rc.local.d/*
    systemctl enable --now rc-local
}

config_create_ap() {
    log_info "config_create_ap.."
    # 系统nftables替换为iptables
    update-alternatives --set iptables /usr/sbin/iptables-legacy
    update-alternatives --set ip6tables /usr/sbin/ip6tables-legacy
    chmod a+x /usr/local/bin/create_ap_adjust.sh
    cp -av /usr/share/doc/hostapd/examples/hostapd.conf /etc/hostapd/hostapd.conf
    cat > /etc/create_ap.conf <<-EOF
CHANNEL=8
GATEWAY=10.0.0.1
WPA_VERSION=2
ETC_HOSTS=0
DHCP_DNS=gateway
NO_DNS=0
NO_DNSMASQ=0
HIDDEN=0
MAC_FILTER=0
MAC_FILTER_ACCEPT=/etc/hostapd/hostapd.accept
ISOLATE_CLIENTS=0
SHARE_METHOD=nat
IEEE80211N=0
IEEE80211AC=0
HT_CAPAB=[HT20][SHORT-GI-20]
VHT_CAPAB=
DRIVER=nl80211
NO_VIRT=0
COUNTRY=CN
FREQ_BAND=2.4
NEW_MACADDR=
DAEMONIZE=0
NO_HAVEGED=0
WIFI_IFACE=wlp5s0
INTERNET_IFACE=enp1s0
SSID=inactivated-$1
PASSPHRASE=12345678
USE_PSK=0
EOF

    # 自动识别wifi网卡名称（多个网卡则获取第一个网卡）
    # WIFIIFACE=$(nmcli device | grep "wifi[ ]" | awk '{print $1}' | head -n1)
    WIRELESS_PRODUCT=$(lshw -c network -class nic 2>/dev/null | awk -F ': ' '/description: Wireless interface/{w=1;next} w&&/product:/{print $2;exit}')
    WIRELESS_VENDOR=$(lshw -c network -class nic 2>/dev/null | awk -F ': ' '/description: Wireless interface/{w=1;next} w&&/vendor:/{print $2;exit}')
    WIRELESS_LOGICAL_NAME=$(lshw -c network -class nic 2>/dev/null | awk -F ': ' '/description: Wireless interface/{w=1;next} w&&/logical name:/&&$2~/^wl/{print $2;exit}')

    echo "Detected WiFi interface: $WIRELESS_LOGICAL_NAME, $WIRELESS_PRODUCT"
    if [ -n "${WIRELESS_LOGICAL_NAME}" ]; then
      sed -i "s/wlp5s0/${WIRELESS_LOGICAL_NAME}/g" /etc/create_ap.conf

      # 检测 5GHz 支持性
      if [[ "${WIRELESS_VENDOR}" == *"Intel"* ]]; then
        ENABLE_5GHz=0    # Intel 无线网卡暂不支持 5GHz AP 模式
      elif [[ "${WIRELESS_PRODUCT}" == *"AR928X"* ]]; then
        ENABLE_5GHz=0    # AR928X 暂不支持 5GHz AP 模式
      elif iw phy $(cat /sys/class/net/${WIRELESS_LOGICAL_NAME}/phy80211/name) info | grep -q "5180 MHz"; then
        ENABLE_5GHz=1
      else
        ENABLE_5GHz=0
      fi

      if [ $ENABLE_5GHz -eq 1 ]; then
        echo "5GHz supported"
        sed -i "s/FREQ_BAND=2.4/FREQ_BAND=5/g" /etc/create_ap.conf
        sed -i "s/CHANNEL=8/CHANNEL=36/g" /etc/create_ap.conf
        sed -i 's/HT_CAPAB=\[HT20\]\[SHORT-GI-20\]/HT_CAPAB=[HT40+][SHORT-GI-40]/g' /etc/create_ap.conf
        sed -i "s/IEEE80211N=1/Ieee80211N=0/g" /etc/create_ap.conf
        sed -i "s/IEEE80211AC=0/IEEE80211AC=1/g" /etc/create_ap.conf
      fi
    fi

#
#    sed -i "s/wlan0/${WIRELESS_LOGICAL_NAME}/g" /etc/hostapd/hostapd.conf
    systemctl disable --now hostapd

    if [ -n "${WIRELESS_LOGICAL_NAME}" ]; then
        systemctl enable create_ap
        systemctl restart create_ap
    fi
}

# 安装driver
install_driver() {
    log_info "install r8168......"
    TARGET_PATH=$(find /lib/modules/$(uname -r)/kernel/drivers/net/ethernet -name realtek -type d)
    if [ "$TARGET_PATH" = "" ]; then
            TARGET_PATH=$(find /lib/modules/$(uname -r)/kernel/drivers/net -name realtek -type d)
    fi
    if [ "$TARGET_PATH" = "" ]; then
            TARGET_PATH=/lib/modules/$(uname -r)/kernel/drivers/net
    fi
    echo
    
    log_info "rmmod r8169"
    /sbin/rmmod r8169 ||:
    log_info "rmmod r8168"
    /sbin/rmmod r8168 ||:
    
    
    log_info "install module"
    #make $@ all 1>>log.txt || exit 1
    cp -av ${ROOT_DIR}/driver/r8168.ko /usr/lib/modules/5.15.137-rt71/kernel/drivers/net/ethernet/realtek/r8168.ko
    depmod -a || true
    module=`ls driver/*.ko | xargs -n1 basename`
    module=${module#src/}
    module=${module%.ko}
    
    if [ "$module" = "" ]; then
            log_error "No driver exists!!!"
            exit 1
    elif [ "$module" != "r8169" ]; then
            if test -e $TARGET_PATH/r8169.ko ; then
                    log_info "Backup r8169.ko"
                    if test -e $TARGET_PATH/r8169.bak ; then
                            i=0
                            while test -e $TARGET_PATH/r8169.bak$i
                            do
                                    i=$(($i+1))
                            done
                            log_info "rename r8169.ko to r8169.bak$i"
                            mv $TARGET_PATH/r8169.ko $TARGET_PATH/r8169.bak$i
                    else
                            log_info "rename r8169.ko to r8169.bak"
                            mv $TARGET_PATH/r8169.ko $TARGET_PATH/r8169.bak
                    fi
            fi
    fi
    
    log_info "DEPMOD $(uname -r)"
    depmod `uname -r`
    log_info "load module $module"
    modprobe $module
    
    is_update_initramfs=n
    distrib_list="ubuntu debian"
    
    if [ -r /etc/debian_version ]; then
            is_update_initramfs=y
    elif [ -r /etc/lsb-release ]; then
            for distrib in $distrib_list
            do
                    /bin/grep -i "$distrib" /etc/lsb-release 2>&1 /dev/null && \
                            is_update_initramfs=y && break
            done
    fi
    
    if [ "$is_update_initramfs" = "y" ]; then
            if which update-initramfs >/dev/null ; then
                    log_info "Updating initramfs. Please wait."
                    update-initramfs -u -k $(uname -r)
            else
                    log_info "update-initramfs: command not found"
                    exit 1
            fi
    fi
    
    log_info "Completed."
}

install_x86_env() {
    log_info "install x86 env"
    sudo dpkg -i x86_env/*.deb
}

clean_srs_and_aubope
dpkg -l xfdesktop4 > /dev/null || install_xubuntu_desktop
copy_files
install_x86_env

# 安装驱动

install_rtl8812au
install_driver

# 系统 patch

fix_hosts
cpu_iso
cpu_affinity
# 虚拟机和docker环境下不执行cpu调频脚本
if [ "$machine_type" = "physical" ]; then cpu_freq; config_sysfsutils; fi
sshd_config
root_auto_login
disable_update_and_apport
optimize_networking_service
disable_bluetooth
disable_useless_service
config_vnc
set_tcp_fin_timeout
config_autousb
config_monitor
config_sysbench
config_exfat
config_7zip
config_create_ap $(date +%s%N | md5sum | cut -c 1-9)
config_logrotate
config_network
config_hdmihotreload
# 虚拟机下不执行安装触屏驱动脚本
if [ "$machine_type" = "physical" ]; then (install_pmLinux && xinput_map && disable_culling); fi
config_rclocal
enable_pulseAudio

# 移除xfce4-panel
apt remove xfce4-panel -y
# 移除屏保应用
apt remove xfce4-screensaver -y

log_info "The ARCS software configuration environment is complete !"
log_info "Successful !"

sync
log_info Installation completed version ${VERSION} arcs environment...
