#!/bin/bash

################################################################################
# File: usbplug.sh
# Date: 03/23-2023
#
# Description:
# Script to mount/unmount usb keys. Called from the udev rule
# "10-automount.rules". Mounts or unmounts the usb key given.
################################################################################

(

   # Make sure only one instance of this script runs at a time
    flock -e 200

    ACTION="$1"
    DEVICE="$2"
    FILESYSTEMS="ext2 ext3 ext4 fat vfat ntfs"


    # Exit on non-zero status from commands
    #set -e
    # Prevent output redirection using `>', `>&', and `<>' from overwriting existing files.
    set -C

    log() {

        /usr/bin/logger -p user.info -t "usbmount[$$]" -- "$1"
    }

    in_list() {
        for v in $2; do
            test "$1" != "$v" || return 0
        done
        return 1
    }

    fstype() {
      eval "$1=`blkid /dev/$2 | sed 's/.* TYPE="\([a-zA-Z0-9]*\)".*/\1/' || true`"
    }

    notify_gui_about_new_usb() {

        if ! echo -e "USB inserted\nquit" | nc 127.0.0.1 29999; then
            log "Failed to notify GUI about insertion of USB key"
        fi
    }

    notify_gui_about_removed_usb() {

        if ! echo -e "USB removed\nquit" | nc 127.0.0.1 29999; then
            log "Failed to notify GUI about removal of USB key"
        fi
    }

    notify_gui_about_unknown_fs() {

        if ! echo -e "USB unknown fs $1\nquit" | nc 127.0.0.1 29999; then
            log "Failed to notify GUI about unknown filesystem"
        fi
    }

    do_cleanup() {

        # Delete stale mountpoints
        for FILE in `find "/media" -maxdepth 1 -mindepth 1 -type d -empty`; do
            if ! mountpoint -q "$FILE"; then
                log "Deleting empty mountpoint $FILE"
                if ! rm -fr "$FILE"; then
                    log "Failed to delete $FILE"
                fi
            fi
        done
    }

    do_mount() {

        MOUNTOPTIONS="rw,sync,dirsync,noexec,nodev,noatime,iocharset=utf8"
        MOUNTOPTIONS_VFAT="rw,sync,dirsync,flush,noexec,nodev,noatime,quiet,dmask=000,fmask=111,iocharset=utf8"

        if [[ ! -b "$1" ]]; then
            log "Device file $1 doesn't exist, or is not a block special file"
            return 1
        fi

        FSTYPE=''
        fstype FSTYPE "$1"
        if ! $FSTYPE; then
            log "Device file $1 isn't a filesystem, exiting."
            return 1
        fi

        if cat /proc/mounts | cut -d' ' -f1 | egrep -q -x "$1"; then
            log "Device $1 already mounted"
            return 1
        fi

        # Create a random mountpoint
        log "my user is $(id)"
        MOUNTPOINT=`mktemp -d -p /media -t urmountpoint_XXXXXX || true`
        if [[ -z "$MOUNTPOINT" ]]; then
            log "Error creating mountpoint"
            return 1
        fi
        log "Created mountpoint $MOUNTPOINT"

        # Create Symlink to mount point in the programs folder
        CNT=0

        #log "mount exec log $(mount -t ext4 -o "$MOUNTOPTIONS" "$1" "$MOUNTPOINT" 2>&1 >/dev/null)"
        #log "systemd-mount exec log $(/usr/bin/systemd-mount --no-block --automount=yes --collect $1 $MOUNTPOINT 2>&1 >/dev/null)"
        if [[ "$FSTYPE" = "vfat" ]]; then
            if ! mount -t vfat -o "$MOUNTOPTIONS_VFAT" "$1" "$MOUNTPOINT"; then
                log "Failed to mount $1 on $MOUNTPOINT"
                log "this is vfat"
                return 1
            fi
        else
            if ! mount -t auto -o "$MOUNTOPTIONS" "$1" "$MOUNTPOINT"; then
                log "Failed to mount $1 on $MOUNTPOINT"
                log "mount -t auto -o "$MOUNTOPTIONS" "$1" "$MOUNTPOINT""
                return 1
            fi
        fi
        log "Mounted device $1 on mountpoint $MOUNTPOINT"

        if ! chown root:root "$MOUNTPOINT"; then
            log "Failed to change ownership on $MOUNTPOINT"
        fi
        if ! chmod 775 "$MOUNTPOINT"; then
            log "Failed to set permissions on $MOUNTPOINT"
        fi

        log "Checking for magic files on $MOUNTPOINT"
        # Check if there are any magic scripts on the key we need to run
        for FILE in `find $MOUNTPOINT -name aubo_magic_*.sh`; do
            if [[ -f "$FILE" ]]; then
                log "Running magic script: $FILE"
                if ! /bin/bash "$FILE" "$MOUNTPOINT"; then
                    log "Error encountered while executing magic script $FILE"
                fi
            fi
        done

        return 0
    }

    do_add() {

        log "do_add $DEVICE"

        do_cleanup

        if [[ -z "$DEVICE" ]]; then
            log "No device supplied, exiting."
            exit 1
        fi

        #This is nessary, as some USB keys is missing a partition table.
        if ! ls "/dev/${DEVICE}1" | egrep -q "${DEVICE}1"; then
            log "This device $DEVICE doesn't seem to have a valid partition table!"
            log "I will try to mount the disk as a partition"
            if ! do_mount "/dev/$DEVICE"; then
                log "Failed to mount disk as partition"
            else
                notify_gui_about_new_usb
            fi
        else
            # Try to mount partititons on the disk
            for DEV in `cat /proc/partitions | awk -F" " '{print $4}' | egrep $DEVICE`; do
                FSTYPE=''
                fstype FSTYPE "$DEV"
                if in_list "$FSTYPE" "$FILESYSTEMS"; then
                    if [[ "$DEV" = "$DEVICE" ]]; then
                        log "This is the disk device, i will not mount it"
                    else
                        if ! do_mount "/dev/$DEV"; then
                            log "Failed to mount $DEV"
                        else
                            notify_gui_about_new_usb
                        fi
                    fi
                fi
            done
        fi

        # Check to see if we actually mounted anything
        if ! cat "/proc/mounts" | cut -d' ' -f1 | egrep -q "/dev/$DEVICE"; then
            log "It seems we failed to mount any devices"
            log "Notifying GUI about unknown FS"
            notify_gui_about_unknown_fs "/dev/$DEVICE"
            exit 1
        fi

    }

    do_remove() {

        log "do_remove $DEVICE"

        if [[ -z "$DEVICE" ]]; then
            log "No device supplied, exiting."
            exit 1
        fi

        for FILE in `cat /proc/mounts | cut -d' ' -f1 | grep "/dev/$DEVICE"`; do
            log "Trying to unmount: $FILE"
            if ! umount -l "$FILE"; then
                log "Error unmounting $FILE"
            else
                do_cleanup
                notify_gui_about_removed_usb
            fi
        done

    }

    do_format() {

        log "Partitioning and formatting $DEVICE"

        if ! fdisk "$DEVICE" < /root/fdisk.script; then
            log "Failed to partition disk"
            exit 1
        fi
        sleep 1
        blockdev --rereadpt "$DEVICE"
        sleep 1
        log "Formatting ${DEVICE}1"
        if ! mkfs.vfat  "${DEVICE}1"; then
            log "Failed to create VFAT filesystem on disk"
        fi
        sleep 1
        sync
        do_mount "${DEVICE}1"
    }

    case "$ACTION" in
        cleanup)
            do_cleanup
            ;;
        add)
            do_add
            ;;
        remove)
            do_remove
            ;;
        format)
            log "do format $DEVICE"
            do_format "$DEVICE"
            ;;
        *)
            echo "Usage: $0 {cleanup|add|remove}"
            exit 1
    esac

    exit 0

) 200>/var/lock/usbplug.lockfile
