#ifndef AUBO_SCOPE_META_TYPES_H
#define AUBO_SCOPE_META_TYPES_H

#include <math.h>
#include <assert.h>
#include <array>
#include <vector>
#include <functional>
#include <string>
#include <initializer_list>
#include <memory>

#include <aubo_caps/domain/data/data_model.h>

namespace arcs {
namespace aubo_scope {

#define STRING_SERIALIZE_ENUM(ENUM_TYPE, ...)                                \
    inline std::string to_string(const ENUM_TYPE &e)                         \
    {                                                                        \
        static_assert(std::is_enum<ENUM_TYPE>::value,                        \
                      #ENUM_TYPE " must be an enum!");                       \
        static const std::pair<ENUM_TYPE, std::string> m[] = __VA_ARGS__;    \
        auto it = std::find_if(                                              \
            std::begin(m), std::end(m),                                      \
            [e](const std::pair<ENUM_TYPE, std::string> &ej_pair) -> bool {  \
                return ej_pair.first == e;                                   \
            });                                                              \
        return ((it != std::end(m)) ? it : std::begin(m))->second;           \
    }                                                                        \
    inline void from_string(const std::string &j, ENUM_TYPE &e)              \
    {                                                                        \
        static_assert(std::is_enum<ENUM_TYPE>::value,                        \
                      #ENUM_TYPE " must be an enum!");                       \
        static const std::pair<ENUM_TYPE, std::string> m[] = __VA_ARGS__;    \
        auto it = std::find_if(                                              \
            std::begin(m), std::end(m),                                      \
            [&j](const std::pair<ENUM_TYPE, std::string> &ej_pair) -> bool { \
                return ej_pair.second == j;                                  \
            });                                                              \
        e = ((it != std::end(m)) ? it : std::begin(m))->first;               \
    }

template <typename T>
class Attribute
{
public:
    Attribute(const std::string &key, T default_value, DataModelPtr model)
        : key_(key), default_value_(default_value), model_(model)
    {
    }

    void setDefault(T v) { default_value_ = v; }

    void reset()
    {
        if constexpr (std::is_enum<T>::value) {
            (model_)->set(key_, to_string(default_value_));
        } else {
            (model_)->set(key_, default_value_);
        }
    }

    void remove() { (model_)->remove(key_); }
    bool exist() { return (model_)->isSet(key_); }
    T get()
    {
        if constexpr (std::is_enum<T>::value) {
            T e;
            from_string((model_)->get(key_, to_string(default_value_)), e);
            return e;
        } else {
            return (model_)->get(key_, default_value_);
        }
    }

    T operator->() { return get(); }

    Attribute &operator=(const T &v)
    {
        if constexpr (std::is_enum<T>::value) {
            (model_)->set(key_, to_string(v));
        } else {
            (model_)->set(key_, v);
        }
    }

    operator T()
    {
        if constexpr (std::is_enum<T>::value) {
            T e;
            from_string((model_)->get(key_, to_string(default_value_)), e);
            return e;
        } else {
            return (model_)->get(key_, default_value_);
        }
    }

private:
    std::string key_;
    T default_value_;
    DataModelPtr model_;
};

#define DECLARE_ATTR(NAME, TYPE, DEF_VALUE)                      \
private:                                                         \
    Attribute<TYPE> NAME##_{ #NAME, (TYPE)(DEF_VALUE), model_ }; \
                                                                 \
public:                                                          \
    TYPE get_##NAME() { return NAME##_; }                        \
    void set_##NAME(TYPE v) { NAME##_ = v; }                     \
    void remove_##NAME() { NAME##_.remove(); }                   \
    bool has_##NAME() { return NAME##_.exist(); }                \
    void reset_##NAME() { NAME##_.reset(); }                     \
    void set_##NAME##_default(TYPE v) { NAME##_.setDefault(v); }

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_META_TYPES_H
