#ifndef AUBO_SCOPE_UNIT_CONVERTOR_H
#define AUBO_SCOPE_UNIT_CONVERTOR_H

#include <math.h>

#include <QObject>
#include <QString>
#include <aubo_caps/domain/system_api.h>

namespace arcs {
namespace aubo_scope {

template <typename T>
class MeasureBase
{
public:
    MeasureBase(double value = 0.) : value_(value) {}

    bool operator==(const MeasureBase &other) { return value_ == other.value_; }

    QString toString(UnitType unit)
    {
        return QString::number(value_) + " " + T::suffix(unit);
    }

    operator double() { return value_; }

protected:
    double value_;
};

class Length : public MeasureBase<Length>
{
public:
    Length(double value = 0.) : MeasureBase(value) {}
    Length(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            value_ = value / 1000.;
            break;
        case UnitType::IMPERIAL:
            value_ = value * 25.4 / 1000.;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("mm", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("in", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_ * 1000.;
            break;
        case UnitType::IMPERIAL:
            return value_ * 1000. / 25.4;
            break;
        }
    }
};

class Speed : public MeasureBase<Speed>
{
public:
    Speed(double value = 0.) : MeasureBase(value) {}
    Speed(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            value_ = value / 1000.;
            break;
        case UnitType::IMPERIAL:
            value_ = value * 25.4 / 1000.;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("mm/s", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("in/s", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_ * 1000.;
            break;
        case UnitType::IMPERIAL:
            return value_ * 1000. / 25.4;
            break;
        }
    }
};

class Acceleration : public MeasureBase<Acceleration>
{
public:
    Acceleration(double value = 0.) : MeasureBase(value) {}
    Acceleration(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            value_ = value / 1000.;
            break;
        case UnitType::IMPERIAL:
            value_ = value * 25.4 / 1000.;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("mm/s²", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("in/s²", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_ * 1000.;
            break;
        case UnitType::IMPERIAL:
            return value_ * 1000. / 25.4;
            break;
        }
        // value_为米,1000毫米; 25.4毫米为一英寸
        return value_;
    }
};

class Angle : public MeasureBase<Angle>
{
public:
    Angle(double value = 0.) : MeasureBase(value) {}
    Angle(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        value_ = value / 180. * M_PI;
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return "°";
            break;
        case UnitType::IMPERIAL:
            return "°";
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_ * 180. / M_PI;
    }
};

class Orientation : public MeasureBase<Orientation>
{
public:
    Orientation(double value = 0.) : MeasureBase(value) {}
    Orientation(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        value_ = value;
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("rad", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("rad", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_;
    }
};

class AngularAcceleration : public MeasureBase<AngularAcceleration>
{
public:
    AngularAcceleration(double value = 0.) : MeasureBase(value) {}
    AngularAcceleration(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        value_ = value / 180. * M_PI;
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("°/s²", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("°/s²", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_ * 180. / M_PI;
    }
};

class AngularSpeed : public MeasureBase<AngularSpeed>
{
public:
    AngularSpeed(double value = 0.) : MeasureBase(value) {}
    AngularSpeed(double value, UnitType unit) { value_ = value / 180. * M_PI; }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("°/s", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("°/s", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_ * 180. / M_PI;
    }
};

class Current : public MeasureBase<Current>
{
public:
    Current(double value = 0.) : MeasureBase(value) {}
    Current(double value, UnitType unit) { value_ = value; }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("A", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("A", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_;
    }
};

class Voltage : public MeasureBase<Voltage>
{
public:
    Voltage(double value = 0.) : MeasureBase(value) {}
    Voltage(double value, UnitType unit) { value_ = value; }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("V", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("V", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_;
    }
};

// 扭力 Nm牛顿米  lb.ft磅英尺
class Torque : public MeasureBase<Torque>
{
public:
    Torque(double value = 0.) : MeasureBase(value) {}
    Torque(double value, UnitType unit)
    {
        // 1(Nm) = 0.737562(lb.ft)
        switch (unit) {
        case UnitType::METRIC:
            value_ = value;
            break;
        case UnitType::IMPERIAL:
            value_ = value * 1.355818;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("Nm", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("lbf.ft", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_;
            break;
        case UnitType::IMPERIAL:
            return value_ / 1.355818;
            break;
        }
        return value_;
    }
};

class Force : public MeasureBase<Force>
{
public:
    Force(double value = 0.) : MeasureBase(value) {}
    Force(double value, UnitType unit)
    {
        // 1(lbf) = 4.45(N)
        switch (unit) {
        case UnitType::METRIC:
            value_ = value;
            break;
        case UnitType::IMPERIAL:
            value_ = value * 4.45;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("N", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("lbf", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_;
            break;
        case UnitType::IMPERIAL:
            return value_ / 4.45;
            break;
        }
        return value_;
    }
};

class Mass : public MeasureBase<Mass>
{
public:
    Mass(double value = 0.) : MeasureBase(value) {}
    Mass(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            value_ = value;
            break;
        case UnitType::IMPERIAL:
            value_ = value * 0.45359237;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("Kg", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("lb", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_;
            break;
        case UnitType::IMPERIAL:
            return value_ / 0.45359237;
            break;
        }
        return value_;
    }
};

class MomentOfInertia : public MeasureBase<MomentOfInertia>
{
public:
    MomentOfInertia(double value = 0.) : MeasureBase(value) {}
    MomentOfInertia(double value, UnitType unit) {}

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("kg·m/s", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("lb in/s", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_;
    }
};

class Moment : public MeasureBase<Moment>
{
public:
    Moment(double value = 0.) : MeasureBase(value) {}
    Moment(double value, UnitType unit)
    {
        // 1(m) = 39.3700787(in)
        // 1(Kg) = 2.20462262185(lb)
        switch (unit) {
        case UnitType::METRIC:
            value_ = value;
            break;
        case UnitType::IMPERIAL:
            value_ = value / 39.3700787 / 2.20462262185;
            break;
        }
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("kg·m/s", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("lb in/s", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return value_;
            break;
        case UnitType::IMPERIAL:
            return value_ * 39.3700787 * 2.20462262185;
            break;
        }
        return value_;
    }
};

class Pressure : public MeasureBase<Pressure>
{
public:
    Pressure(double value = 0.) : MeasureBase(value) {}
    Pressure(double value, UnitType unit) {}

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("kg·m/s", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("lb in/s", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_;
    }
};

class Time : public MeasureBase<Time>
{
public:
    Time(double value = 0.) : MeasureBase(value) {}
    Time(double value, UnitType unit) {}

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("s", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("s", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_;
    }
};

class TinyTime : public MeasureBase<Time>
{
public:
    TinyTime(double value = 0.) : MeasureBase(value) {}
    TinyTime(double value, UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }

        value_ = value / 1000.;
    }

    static QString suffix(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            return QObject::tr("ms", "unit_suffix");
            break;
        case UnitType::IMPERIAL:
            return QObject::tr("ms", "unit_suffix");
            break;
        }
    }

    double getAs(UnitType unit)
    {
        switch (unit) {
        case UnitType::METRIC:
            break;
        case UnitType::IMPERIAL:
            break;
        }
        return value_ * 1000.;
    }
};

class Power : public MeasureBase<Power>
{
public:
    Power(double value = 0.) : MeasureBase(value) {}
    Power(double value, UnitType unit) : MeasureBase(value) {}

    static QString suffix(UnitType unit)
    {
        return QObject::tr("W", "unit_suffix");
    }

    double getAs(UnitType unit) { return value_; }
};

class Frequency : public MeasureBase<Frequency>
{
public:
    Frequency(double value = 0.) : MeasureBase(value) {}
    Frequency(double value, UnitType unit) : MeasureBase(value) {}

    static QString suffix(UnitType unit)
    {
        return QObject::tr("Hz", "unit_suffix");
    }

    double getAs(UnitType unit) { return value_; }
};

} // namespace aubo_scope
} // namespace arcs
#endif
