#ifndef AUBO_CAPS_PP_AUBO_FUNCTION_H
#define AUBO_CAPS_PP_AUBO_FUNCTION_H

#include <QString>
#include <QRegExp>
#include <QVector>

namespace arcs {
namespace aubo_scope {

class Utils
{
public:
    // 多字节字符 转 十六进制 Unicode 字符串
    static QString multibyteToUnicode(const QString &src)
    {
        QString dest;

        for (auto &c : src) {
            auto unicode = c.unicode();

            // 不管什么语言，处理逻辑都是一样的。只有一个英文不需要处理
            // 0x007F以内是ASCII,确认不需做转码处理

            if (unicode < 0x7f) {
                dest.append(c);
            } else {
                dest.append(QString("u%1").arg(unicode, 4, 16, QChar('0')));
            }
        }

        return dest;
    }

    inline static QString multibyteToUnicode(const char *src)
    {
        return multibyteToUnicode(QString(src));
    }

    inline static std::string multibyteToUnicode(const std::string &src)
    {
        return multibyteToUnicode(QString::fromStdString(src)).toStdString();
    }

    // 十六进制 Unicode 字符串 转 多字节字符
    static QString unicodeToMultibyte(const QString &src)
    {
        QString dest;

        // 十六进制 unicode 字符串 替换为 多字节字符
        // "u53d8u91cf_0" -> "变量_0"
        for (int i = 0; i < src.size(); i++) {
            // 'u' 开头
            if (src.at(i) == 'u') {
                auto is_hex = false;
                auto hex = src.midRef(i + 1, 4).toUShort(&is_hex, 16);

                // 后面 4 位是十六进制字符串
                if (is_hex) {
                    dest.append(QChar(hex));
                    i += 4;
                    continue;
                }
            }

            dest.append(src.at(i));
        }

        return dest;
    }

    inline static QString unicodeToMultibyte(const char *src)
    {
        return unicodeToMultibyte(QString(src));
    }

    inline static std::string unicodeToMultibyte(const std::string &src)
    {
        return unicodeToMultibyte(QString::fromStdString(src)).toStdString();
    }

    /**
     * @brief compareVersion 比较版本大小
     * @param v1 版本
     * @param v2 版本
     * @return 若版本 v1 < v2 则返回 -1;
     *  v1 == v2 返回 0;
     *  v1 > v2 返回 1;
     *  版本号格式不符，返回 -2;
     *
     * 规则：0.28.4-alpha.100 < 0.28.4-rc.1 < 0.28.4
     * 版本格式：{MAJOR}.{MINOR}.{REVISION}-{PRE-RELEASE}
     *  {MAJOR}：主板本号.
     *  {MINOR}：次版本号.
     *  {REVISION}：修订号.
     *  {PRE-RELEASE}：预发布版本号，最终版本无此部分.
     */
    static int compareVersion(const QString &v1, const QString &v2)
    {
        auto v1_nums = v1.splitRef(".");
        auto v2_nums = v2.splitRef(".");

        // 版本完全相同
        if (v1 == v2) {
            return 0;
        }

        // 至少有 {MAJOR}.{MINOR}.{REVISION} 三个版本号
        if (v1_nums.size() < 3 || v2_nums.size() < 3) {
            return -2;
        }

        // 比较 {MAJOR}
        if (v1_nums.at(0) != v2_nums.at(0)) {
            return v1_nums.at(0) < v2_nums.at(0) ? -1 : 1;
        }

        // 比较 {MINOR}
        if (v1_nums.at(1) != v2_nums.at(1)) {
            return v1_nums.at(1) < v2_nums.at(1) ? -1 : 1;
        }

        // 比较 {REVISION}，同时附带比较 alpha < rc
        // 没有预发布版本的是最终版本，版本最高
        auto v1_revision = v1_nums.at(2).split("-");
        auto v2_revision = v1_nums.at(2).split("-");
        if (v1_revision.at(0) != v2_revision.at(0)) { // 修订号比较
            return v1_revision.at(0) < v2_revision.at(0) ? -1 : 1;
        } else if (v1_revision.size() != v2_revision.size()) {
            // 是否都有预发布版本，没有的最大
            return v1_revision.size() < v2_revision.size() ? 1 : -1;
        } else { // 预发布版本比较
            const QStringList revision_sort = { "alpha", "rc" };
            return revision_sort.indexOf(v1_revision.at(1).toString()) <
                           revision_sort.indexOf(v2_revision.at(1).toString())
                       ? -1
                       : 1;
        }

        // 比较 {PRE-RELEASE}
        if (v1_nums.at(3) != v2_nums.at(3)) {
            return v1_nums.at(3) < v2_nums.at(3) ? -1 : 1;
        }

        return 0;
    }

    inline static int compareVersion(const std::string &v1,
                                     const std::string &v2)
    {
        return compareVersion(QString::fromStdString(v1),
                              QString::fromStdString(v2));
    }

private:
    Utils() {}
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_CAPS_PP_AUBO_FUNCTION_H
