#ifndef AUBO_SCOPE_CLASS_FORWARD_H
#define AUBO_SCOPE_CLASS_FORWARD_H

#include <memory>

#if defined(ARCS_ABI_EXPORT)
#elif defined(ARCS_PLATFORM_WINDOWS)
#define ARCS_ABI_EXPORT __declspec(dllexport)
#define ARCS_ABI_IMPORT __declspec(dllimport)
#define ARCS_ABI_LOCAL
#elif defined(ARCS_HAVE_VISIBILITY_ATTRIBUTE)
#define ARCS_ABI_EXPORT __attribute__((visibility("default")))
#define ARCS_ABI_IMPORT __attribute__((visibility("default")))
#define ARCS_ABI_LOCAL  __attribute__((visibility("hidden")))
#else
#define ARCS_ABI_EXPORT
#define ARCS_ABI_IMPORT
#define ARCS_ABI_LOCAL
#endif

/**
 * @def ARCS_DELCARE_PTR
 * Macro that given a Name and a Type declares the following types:
 * - ${Name}Ptr            = shared_ptr<${Type}>
 * - ${Name}ConstPtr       = shared_ptr<const ${Type}>
 * - ${Name}WeakPtr        = weak_ptr<${Type}>
 * - ${Name}ConstWeakPtr   = weak_ptr<const ${Type}>
 * - ${Name}UniquePtr      = unique_ptr<${Type}>
 * - ${Name}ConstUniquePtr = unique_ptr<const ${Type}>
 *
 * For best portability the exact type of shared_ptr declared by the macro
 * should be considered to be an implementation detail, liable to change in
 * future releases.
 */

#define ARCS_DECLARE_PTR(Name, Type)                      \
    typedef std::shared_ptr<Type> Name##Ptr;              \
    typedef std::shared_ptr<const Type> Name##ConstPtr;   \
    typedef std::weak_ptr<Type> Name##WeakPtr;            \
    typedef std::weak_ptr<const Type> Name##ConstWeakPtr; \
    typedef std::unique_ptr<Type> Name##UniquePtr;        \
    typedef std::unique_ptr<const Type> Name##ConstUniquePtr

/**
 * @def ARCS_DELCARE_PTR_MEMBER
 * The macro defines the same typedefs as ARCS_DECLARE_PTR, but shortens the
 * new names to their suffix.
 *
 * This can be used to create `Classname::Ptr` style names, but in most
 * situations in ARCS's codebase, ARCS_CLASS_FORWARD and ARCS_DECLARE_PTR
 * should be preferred.
 */

#define ARCS_DECLARE_PTR_MEMBER(Type)               \
    typedef std::shared_ptr<Type> Ptr;              \
    typedef std::shared_ptr<const Type> ConstPtr;   \
    typedef std::weak_ptr<Type> WeakPtr;            \
    typedef std::weak_ptr<const Type> ConstWeakPtr; \
    typedef std::unique_ptr<Type> UniquePtr;        \
    typedef std::unique_ptr<const Type> ConstUniquePtr

/**
 * @def ARCS_CLASS_FORWARD
 * Macro that forward declares a class and defines the respective smartpointers
 * through ARCS_DECLARE_PTR.
 */

#define ARCS_CLASS_FORWARD(C) \
    class C;                  \
    ARCS_DECLARE_PTR(C, C)

/**
 * @def ARCS_STRUCT_FORWARD
 * Like ARCS_CLASS_FORWARD, but forward declares the type as a struct
 * instead of a class.
 */
#define ARCS_STRUCT_FORWARD(C) \
    struct C;                  \
    ARCS_DECLARE_PTR(C, C)

#endif
