#ifndef AUBO_SCOPE_GRIP_FORCE_CAPABILITY_H
#define AUBO_SCOPE_GRIP_FORCE_CAPABILITY_H

#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(GripForceCapability);

/**
 * <p>
 * This interface represents a registered force capability for a gripper which
 * supports gripping using a user configurable force.
 * </p>
 *
 * Through this interface, it is possible to adjust the range and default value
 * for the force after the capability has been registered. This is typically not
 * applicable for most grippers, but can be relevant when the range depends on
 * another custom gripper setting (which can be configured by the user).
 */
class GripForceCapability
{
public:
    virtual ~GripForceCapability() = 0;

    /**
     * <p>
     * Update the range and default value for the force supported by the
     * gripper.
     * </p>
     *
     * Updating the range could result in an already entered value in the
     * configuration of a gripper program node being outside the range. This
     * will make the given program node undefined, and force the end user to
     * adjust the value to be valid again.
     *
     * @param minForce minimum force supported by the gripper
     * @param maxForce maximum force supported by the gripper
     * @param defaultGripForce default value for the force for a grip action.
     * This value is used for, e.g. performing a "default" grip action using the
     * toolbar
     * @param unit the unit for all specified values, <code>null</code>
     * @throws InvalidCapabilityRange if <code>minForce</code> > <code>maxForce}
     * or <code>defaultGripForce</code> is outside the range defined by
     * <code>minForce</code> and <code>maxForce</code>
     */
    virtual void updateCapability(double minForce, double maxForce,
                                  double defaultGripForce) = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
