#ifndef AUBO_SCOPE_GRIP_VACUUM_CAPABILITY_H
#define AUBO_SCOPE_GRIP_VACUUM_CAPABILITY_H

#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(GripVacuumCapability);

/**
 * <p>
 * This interface represents a registered vacuum capability for a gripper which
 * supports gripping using a user configurable vacuum level.
 * </p>
 *
 * Through this interface, it is possible to adjust the range and default value
 * for the vacuum level after the capability has been registered. This is
 * typically not applicable for most grippers, but can be relevant when the
 * range depends on another custom gripper setting (which can be configured by
 * the user).
 */
class GripVacuumCapability
{
public:
    virtual ~GripVacuumCapability() = 0;

    /**
     * <p>
     * Update the range and default value for the vacuum level supported by the
     * gripper.
     * </p>
     *
     * <p>
     * Updating the range could result in an already entered value in the
     * configuration of a gripper program node being outside the range. This
     * will make the given program node undefined, and force the end user to
     * adjust the value to be valid again.
     * </p>
     *
     * If the gripper works with absolute pressure, specify a positive range for
     * the supported vacuum level. If the gripper works with relative pressure
     * (vacuum), specify a negative range for the supported vacuum level.
     *
     * @param minVacuum minimum vacuum level supported by the gripper
     * @param maxVacuum maximum vacuum level supported by the gripper
     * @param defaultGripVacuum default value for the vacuum level for a grip
     * action. This value is used for, e.g. performing a "default" grip action
     * using the toolbar
     * @param unit the unit for all specified values, not <code>null</code>
     * @throws InvalidCapabilityRange if <code>minVacuum</code> >
     * <code>maxVacuum</code> or <code>defaultGripVacuum</code> is outside the
     * range defined by <code>minVacuum</code> and <code>maxVacuum</code>.
     */
    virtual void updateCapability(double minVacuum, double maxVacuum,
                                  double defaultGripVacuum) = 0;
};
} // namespace aubo_scope
} // namespace arcs

#endif
