#ifndef AUBO_SCOPE_REGISTERED_CAPABILITIES_H
#define AUBO_SCOPE_REGISTERED_CAPABILITIES_H

#include <aubo_caps/contribution/driver/gripper/gripper_api_provider.h>
#include <aubo_caps/contribution/driver/gripper/gripper_configuration.h>
#include <aubo_caps/contribution/driver/gripper/gripper_contribution.h>
#include <aubo_caps/contribution/driver/gripper/capability/grip_force_capability.h>
#include <aubo_caps/contribution/driver/gripper/capability/grip_vacuum_capability.h>
#include <aubo_caps/contribution/driver/gripper/capability/speed_capability.h>
#include <aubo_caps/contribution/driver/gripper/capability/width_capability.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(RegisteredCapabilities);

/**
 * <p>
 * Each of the individual grippers/zones available in a multi-gripper device
 * supports all capabilities that has been registered, i.e. all the individual
 * grippers have the exact same set of capabilities. This interface gives access
 * to all registered capabilities, such as the width capability (see {@link
 * WidthCapability}), speed capability (see
 * {@link SpeedCapability}), etc. This enables dynamic adjustment of capability
 * properties, including the value range and default values, exclusively per
 * individual gripper (independently of the other grippers).
 * </p>
 *
 * <p>
 * This interface is only relevant if the multi-gripper capability has been
 * registered (using the method
 * {@link GripperCapabilities#registerWidthCapability(double, double, double,
 * double, Length.Unit)}).
 * </p>
 *
 * <p>
 * Each capability <b>must</b> be registered using the {@link
 * GripperCapabilities} interface (when
 * {@link GripperContribution#configureGripper(GripperConfiguration,
 * GripperAPIProvider)} is called) before they can be accessed using this
 * interface.
 * </p>
 *
 * <b>NOTE:</b> If an update of the properties of a capability needs to apply
 * for all individual grippers, it is more convenient to perform the adjustment
 * simultaneously for all grippers using the capability instance (e.g.
 * {@link WidthCapability}) returned when the capability was registered (with
 * the {@link GripperCapabilities} interface).
 */
class RegisteredCapabilities
{
public:
    virtual ~RegisteredCapabilities() = default;

    /**
     * <p>
     * Access the width capability interface which allows for dynamic
     * adjustments of the capability properties exclusively for this
     * specific individual gripper (independently of the other grippers),
     * including the value range and default value.
     * </p>
     *
     * This is only valid, if the width capability has been registered
     * (using the method
     * {@link GripperCapabilities#registerWidthCapability(double, double,
     * double, double, Length.Unit)}).
     *
     * @return The width capability for this individual gripper
     * @throws UnsupportedOperationException if the gripper has not
     * registered the width capability
     */
    virtual WidthCapabilityPtr getWidthCapability() = 0;

    /**
     * <p>
     * Access the grip force capability interface which allows for dynamic
     * adjustments of the capability properties exclusively for this specific
     * individual gripper (independently of the other grippers), including the
     * value range and default value.
     * </p>
     *
     *
     * This is only valid, if the force capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerGrippingForceCapability(double,
     * double, double, Force.Unit)}).
     *
     * @return The grip force capability for this individual gripper
     * @throws UnsupportedOperationException if the gripper has not registered
     * the force capability
     */
    virtual GripForceCapabilityPtr getGripForceCapability() = 0;

    /**
     * <p>
     * Access the speed capability interface which allows for dynamic
     * adjustments of the capability properties exclusively for this specific
     * individual gripper (independently of the other grippers), including the
     * value range and default value.
     * </p>
     *
     * This is only valid, if the speed capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerSpeedCapability(double, double,
     * double, double, Speed.Unit)}).
     *
     * @return The speed capability for this individual gripper
     * @throws UnsupportedOperationException if the gripper has not registered
     * the speed capability
     */
    virtual SpeedCapabilityPtr getSpeedCapability() = 0;

    /**
     * <p>
     * Access the grip vacuum capability interface which allows for dynamic
     * adjustments of the capability properties exclusively for this specific
     * individual gripper (independently of the other grippers), including the
     * value range and default value.
     * </p>
     *
     * This is only valid, if the vacuum capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerGrippingVacuumCapability(double,
     * double, double, Pressure.Unit)}).
     *
     * @return The grip vacuum capability for this individual gripper
     * @throws UnsupportedOperationException if the gripper has not registered
     * the vacuum capability
     */
    virtual GripVacuumCapabilityPtr getGripVacuumCapability() = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
