#ifndef AUBO_SCOPE_SPEED_CAPABILITY_H
#define AUBO_SCOPE_SPEED_CAPABILITY_H

#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(SpeedCapability);

/**
 * <p>
 * This interface represents a registered speed capability for a gripper which
 * supports operating using a user configurable speed.
 * </p>
 *
 * Through this interface, it is possible to adjust the range and default values
 * for the speed after the capability has been registered. This is typically not
 * applicable for most grippers, but can be relevant when the range depends on
 * another custom gripper setting (which can be configured by the user).
 */
class SpeedCapability
{
public:
    virtual ~SpeedCapability() = default;

    /**
     * <p>
     * Update the range and default values for the speed supported by the
     * gripper.
     * </p>
     *
     * Updating the range could result in an already entered value in the
     * configuration of a gripper program node being outside the range. This
     * will make the given program node undefined, and force the end user to
     * adjust the value to be valid again.
     *
     * @param minSpeed minimum operating speed supported by the gripper
     * @param maxSpeed maximum operating speed supported by the gripper
     * @param defaultGripSpeed default value for the speed for a grip action.
     * This value is used for, e.g. performing a "default" grip action using the
     * toolbar
     * @param defaultReleaseSpeed default value for the speed for a release
     * action. This value is used for, e.g. performing a "default" release
     * action using the toolbar
     * @param unit the unit for all specified values, not <code>null</code>
     * @throws InvalidCapabilityRange if <code>minSpeed</code> >
     * <code>maxSpeed</code>
     * @throws InvalidCapabilityRange if <code>defaultGripSpeed</code> or
     * <code>defaultReleaseSpeed</code> is outside the range defined by
     * <code>minSpeed</code> and <code>maxSpeed</code>.
     */
    virtual void updateCapability(double minSpeed, double maxSpeed,
                                  double defaultGripSpeed,
                                  double defaultReleaseSpeed) = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
