#ifndef AUBO_SCOPE_WIDTH_CAPABILITY_H
#define AUBO_SCOPE_WIDTH_CAPABILITY_H

#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(WidthCapability);

/**
 * <p>
 * This interface represents a registered width/position capability for a
 * gripper which supports moving to a user configurable position (open/close to
 * a configurable width).
 * </p>
 *
 * Through this interface, it is possible to adjust the range and default values
 * for the width after the capability has been registered. This is typically not
 * applicable for most grippers, but can be relevant when the range depends on
 * another custom gripper setting (which can be configured by the user).
 */
class WidthCapability
{
public:
    virtual ~WidthCapability() = default;

    /**
     * <p>
     * Update the range and default values for the width supported by the
     * gripper.
     * </p>
     *
     * Updating the range could result in an already entered value in the
     * configuration of a gripper program node being outside the range. This
     * will make the given program node undefined, and force the end user to
     * adjust the value to be valid again.
     *
     * @param minWidth minimum width supported by the gripper
     * @param maxWidth maximum width supported by the gripper
     * @param defaultGripWidth default value for the width for a grip
     * action. This value is used for, e.g. performing a "default" grip
     * action using the toolbar
     * @param defaultReleaseWidth default value for the width for a release
     * action. This value is used for, e.g. performing a "default" release
     * action using the toolbar
     * @param unit the unit for all specified values, not <code>null</code>
     * @throws InvalidCapabilityRange if <code>minWidth} > <code>maxWidth</code>
     * @throws InvalidCapabilityRange if <code>defaultGripWidth</code> or
     * <code>defaultReleaseWidth</code> is outside the range defined by
     * <code>minWidth</code> and <code>maxWidth</code>.
     */
    virtual void updateCapability(double minWidth, double maxWidth,
                                  double defaultGripWidth,
                                  double defaultReleaseWidth) = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
