#ifndef AUBO_SCOPE_GRIP_ACTION_PARAMETERS_H
#define AUBO_SCOPE_GRIP_ACTION_PARAMETERS_H

#include <aubo_caps/domain/script/script_writer.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(GripActionParameters);

/**
 * <p>
 * This interface represents parameters for a gripping action defined/configured
 * by the end user.
 * </p>
 *
 * These parameters are used when generating script code for a gripping action
 * (see
 * {@link GripperContribution#generateGripActionScript(ScriptWriter,
 * GripActionParameters)}).
 */
class GripActionParameters
{
public:
    virtual ~GripActionParameters() = default;

    /**
     * <p>
     * Get the selected force to be used the gripping action.
     * </p>
     *
     * This is only valid, if the force capability has been registered
     * (using the method
     * {@link GripperCapabilities#registerGrippingForceCapability(double,
     * double, double, Force.Unit)}).
     *
     * @return The force to use for the gripping action
     * @throws UnsupportedOperationException if the gripper has not
     * registered the force capability
     */
    virtual double getForce() = 0;

    /**
     * <p>
     * Get the selected width to be used for the gripping action.
     * </p>
     *
     * This is only valid, if the width capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerWidthCapability(double, double,
     * double, double, Length.Unit)}).
     *
     * @return The width to use for the gripping action
     * @throws UnsupportedOperationException if the gripper has not registered
     * the width capability
     */
    virtual double getWidth() = 0;

    /**
     * <p>
     * Get the selected vacuum level to be used for the gripping action.
     * </p>
     *
     * This is only valid, if the vacuum capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerGrippingVacuumCapability(double,
     * double, double, Pressure.Unit)}).
     *
     * @return The vacuum level to use for the gripping action
     * @throws UnsupportedOperationException if the gripper has not registered
     * the vacuum capability
     */
    virtual double getVacuum() = 0;

    /**
     * <p>
     * Get the selected speed to be used for the gripping action.
     * </p>
     *
     * This is only valid, if the speed capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerSpeedCapability(double, double,
     * double, double, Speed.Unit)}).
     *
     * @return The speed to use for the gripping action
     * @throws UnsupportedOperationException if the gripper has not registered
     * the speed capability
     */
    virtual double getSpeed() = 0;

    /**
     * <p>
     * Use this method to determine whether the grip detection option in the
     * Gripper program node is on or off.
     * </p>
     *
     * <p>
     * Note that this method will report that grip detection is off, when script
     * code is to be generated due to the user testing a grip configuration of a
     * gripper program or operating the gripper using the toolbar.
     * </p>
     *
     * <p>
     * This method should only be called, if the grip detected feedback
     * capability has been registered (using
     * {@link
     * GripperFeedbackCapabilities#registerGripDetectedCapability(ScriptCodeGenerator)}).
     * </p>
     *
     * <p>
     * The purpose of this method is to ensure that the appropriate script code
     * for performing a grip action is generated when the end user has specified
     * a new payload value. The user-defined payload value will be applied by
     * AuboScope immediately after the generated grip action script code has
     * finished executing. Depending on the enablement state of grip detection,
     * the generated script should have the following properties: <ul> <li>
     *          <i>Grip detection off:</i> The script code should not finish
     * earlier than when it is appropriate to apply the new payload value (i.e.
     * when the object has been gripped). This typically means that the script
     * code should wait a time period corresponding to closing the gripper's
     * "fingers" fully (if the gripper is vacuum operated then long enough to
     * achieve some level of vacuum).
     *      </li>
     *      <li>
     * 	        <i>Grip detection on:</i> The payload will only be set if the
     * grip detection was successful, i.e. it will be applied immediately after
     * a grip was detected. This means that it is not necessary for the script
     * code to wait (since the built-in detection timeout functionality will
     * handle the waiting part).
     * 	     </li>
     *  </ul>
     * </p>
     *
     * @return <code>true</code> if the end user has enabled grip detection, and
     * <code>false</code> if grip detection is disabled.
     * @throws UnsupportedOperationException if the gripper has not registered
     * the grip detected feedback capability
     */
    virtual bool isGripDetectionEnabled() = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
