#ifndef AUBO_SCOPE_GRIPPER_CONFIGURATION_H
#define AUBO_SCOPE_GRIPPER_CONFIGURATION_H

#include <functional>
#include <aubo_caps/domain/script/script_writer.h>
#include <aubo_caps/contribution/driver/gripper/capability/width_capability.h>
#include <aubo_caps/contribution/driver/gripper/capability/speed_capability.h>
#include <aubo_caps/contribution/driver/gripper/capability/grip_force_capability.h>
#include <aubo_caps/contribution/driver/gripper/capability/grip_vacuum_capability.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(GripperConfiguration);

/**
 * This interface provides access to register or setup various properties and
 * capabilities of a gripper. It is optional to register these capabilities and
 * properties.
 */
class GripperConfiguration
{
public:
    virtual ~GripperConfiguration() = default;

    /**
     * Register a width/position capability for a gripper that supports moving
     * to a user configurable position (open/close to a configurable width).
     *
     * @param minWidth The minimum width supported by the gripper
     * @param maxWidth The maximum width supported by the gripper
     * @param defaultGripWidth default value for the width for a grip action.
     * This value is used for, e.g. performing a "default" grip action using the
     * toolbar
     * @param defaultReleaseWidth default value for the width for a release
     * action. This value is used for, e.g. performing a "default" release
     * action using the toolbar
     * @param unit the unit for all specified values, not <code>null</code>
     * @return a width capability allowing for dynamic adjustments of the
     * capability properties including the value range and default values
     * @throws InvalidCapabilityRange if <code>minWidth</code> >
     * <code>maxWidth</code>
     * @throws InvalidCapabilityRange if <code>defaultGripWidth</code> or
     * <code>defaultReleaseWidth</code> is outside the range defined by
     * <code>minWidth</code> and <code>maxWidth</code>.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideGripperConfigurationPhase if this method is called
     * at the wrong time by a AuboCap implementing the {@link
     * GripperContribution} interface, i.e. called outside the scope of the
     * configuration of the gripper.
     * @throws CalledOutsideMethodScope if this method is called at the wrong
     * time by a (regular) AuboCap which has registered its program node
     * contribution/service as conformant with a AuboScope gripper (using
     *         {@link GripperRegistrationManager#registerAsGripper(Class)}),
     * i.e. called outside the scope of the call to either {@link
     * SwingProgramNodeService#configureContribution(ContributionConfiguration)}
     * (for a Swing-based AuboCap) or {@link
     * ProgramNodeServiceConfigurable#configureContribution(ProgramNodeConfiguration)}
     * (for a HTML-based AuboCap).
     */
    virtual WidthCapabilityPtr registerWidthCapability(
        double minWidth, double maxWidth, double defaultGripWidth,
        double defaultReleaseWidth) = 0;

    /**
     * Register a force capability for a gripper that supports gripping using a
     * user configurable force.
     *
     * @param minForce minimum force supported by the gripper
     * @param maxForce maximum force supported by the gripper
     * @param defaultGripForce default value for the force for a grip action.
     * This value is used for, e.g. performing a "default" grip action using the
     * toolbar
     * @param unit the unit for all specified values, <code>null</code>
     * @return a force capability allowing for dynamic adjustments of the
     * capability properties including the value range and default value
     * @throws InvalidCapabilityRange if <code>minForce</code> >
     * <code>maxForce</code> or <code>defaultGripForce</code> is outside the
     * range defined by <code>minForce</code> and <code>maxForce</code>.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideGripperConfigurationPhase if this method is called
     * at the wrong time by a AuboCap implementing the {@link
     * GripperContribution} interface, i.e. called outside the scope of the
     * configuration of the gripper.
     * @throws CalledOutsideMethodScope if this method is called at the wrong
     * time by a (regular) AuboCap which has registered its program node
     * contribution/service as conformant with a AuboScope gripper (using
     *         {@link GripperRegistrationManager#registerAsGripper(Class)}),
     * i.e. called outside the scope of the call to either {@link
     * SwingProgramNodeService#configureContribution(ContributionConfiguration)}
     * (for a Swing-based AuboCap) or {@link
     * ProgramNodeServiceConfigurable#configureContribution(ProgramNodeConfiguration)}
     * (for a HTML-based AuboCap).
     */
    virtual GripForceCapabilityPtr registerGrippingForceCapability(
        double minForce, double maxForce, double defaultGripForce) = 0;

    /**
     * Register a speed capability for a gripper that supports operating using a
     * user configurable speed.
     *
     * @param minSpeed minimum speed supported by the gripper
     * @param maxSpeed maximum speed supported by the gripper
     * @param defaultGripSpeed default value for the speed for a grip action.
     * This value is used for, e.g. performing a "default" grip action using the
     * toolbar
     * @param defaultReleaseSpeed default value for the speed for a grip action.
     * This value is used for, e.g. performing a "default" release action using
     * the toolbar
     * @param unit the unit for all specified values, not <code>null</code>
     * @return a speed capability allowing for dynamic adjustments of the
     * capability properties including the value range and default values
     * @throws InvalidCapabilityRange if <code>minSpeed</code> >
     * <code>maxSpeed</code>
     * @throws InvalidCapabilityRange if <code>defaultGripSpeed</code> or
     * defaultReleaseSpeed</code> is outside the range defined by
     * <code>minSpeed</code> and <code>maxSpeed</code>.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideGripperConfigurationPhase if this method is called
     * at the wrong time by a AuboCap implementing the {@link
     * GripperContribution} interface, i.e. called outside the scope of the
     * configuration of the gripper.
     * @throws CalledOutsideMethodScope if this method is called at the wrong
     * time by a (regular) AuboCap which has registered its program node
     * contribution/service as conformant with a AuboScope gripper (using
     *         {@link GripperRegistrationManager#registerAsGripper(Class)}),
     * i.e. called outside the scope of the call to either {@link
     * SwingProgramNodeService#configureContribution(ContributionConfiguration)}
     * (for a Swing-based AuboCap) or {@link
     * ProgramNodeServiceConfigurable#configureContribution(ProgramNodeConfiguration)}
     * (for a HTML-based AuboCap).
     */
    virtual SpeedCapabilityPtr registerSpeedCapability(
        double minSpeed, double maxSpeed, double defaultGripSpeed,
        double defaultReleaseSpeed) = 0;

    /**
     * <p>
     * Register a vacuum capability for a gripper that supports gripping using a
     * user configurable vacuum level.
     * </p>
     *
     * If the gripper works with absolute pressure, specify a positive range for
     * the supported vacuum level. If the gripper works with relative pressure
     * (vacuum), specify a negative range for the supported vacuum level.
     *
     * @param minVacuum minimum vacuum level supported by the gripper
     * @param maxVacuum maximum vacuum level supported by the gripper
     * @param defaultGripVacuum default value for the vacuum level for a grip
     * action. This value is used for, e.g. performing a "default" grip action
     * using the toolbar
     * @param unit the unit for all specified values, not <code>null</code>
     * @return a vacuum capability allowing for dynamic adjustments of the
     * capability properties including the value range and default value
     * @throws InvalidCapabilityRange if <code>minVacuum</code> >
     * <code>maxVacuum</code> or <code>defaultGripVacuum</code> is outside the
     * range defined by <code>minVacuum</code> and <code>maxVacuum</code>.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideGripperConfigurationPhase if this method is called
     * at the wrong time by a AuboCap implementing the {@link
     * GripperContribution} interface, i.e. called outside the scope of the
     * configuration of the gripper.
     * @throws CalledOutsideMethodScope if this method is called at the wrong
     * time by a (regular) AuboCap which has registered its program node
     * contribution/service as conformant with a AuboScope gripper (using
     *         {@link GripperRegistrationManager#registerAsGripper(Class)}),
     * i.e. called outside the scope of the call to either {@link
     * SwingProgramNodeService#configureContribution(ContributionConfiguration)}
     * (for a Swing-based AuboCap) or {@link
     * ProgramNodeServiceConfigurable#configureContribution(ProgramNodeConfiguration)}
     * (for a HTML-based AuboCap).
     */
    virtual GripVacuumCapabilityPtr registerGrippingVacuumCapability(
        double minVacuum, double maxVacuum, double defaultGripVacuum) = 0;

    /**
     * <p>
     * Register this feedback capability if the gripper can inform AuboScope
     * whether it has detected that an object has been gripped, after a grip
     * action has been triggered. This will enable the end user to configure the
     * actions to take when the gripper has detected that an object has been
     * gripped and/or when the grip action timed out.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for determining if a grip was detected. The
     * return value of the script code <b>must</b> be a boolean, i.e. 'True' or
     * 'False'.
     * </p>
     *
     * <b>Note:</b> When this capability has been registered, the
     * {@link GripperContribution#generateGripActionScript(ScriptWriter,
     * GripActionParameters)} method should generate different grip action
     * script code depending on whether the end user has enabled or disabled the
     * grip detection option (in the Gripper program node). For more details,
     * see Javadoc for
     * {@link GripActionParameters#isGripDetectionEnabled()}.
     *
     * @param scriptCodeGenerator script generator that generates the script
     * code for determining if a grip was detected. When the script code needs
     * to be generated, the method
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} will be called
     * (by AuboScope). The resulting script code will be embedded in a script
     * function and <b>must</b> return a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideGripperConfigurationPhase if this method is called
     * at the wrong time, i.e. outside the scope of the {@link
     * GripperContribution#configureGripper(GripperConfiguration,
     * GripperAPIProvider)} method.
     */
    virtual void registerGripDetectedCapability(
        std::function<void(ScriptWriterPtr scriptWriter)>
            scriptCodeGenerator) = 0;

    /**
     * <p>
     * Register this feedback capability if the gripper can inform AuboScope
     * whether it has detected an object has been released, after a release
     * action has been triggered. This will enable the end user to configure the
     * actions to take when the gripper has detected that an object has been
     * released and/or when the release action timed out.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for determining if a release was detected.
     * The return value of the script code <b>must</b> be a boolean, i.e. 'True'
     * or 'False'.
     * </p>
     *
     * <b>Note:</b> When this capability has been registered, the
     * {@link GripperContribution#generateReleaseActionScript(ScriptWriter,
     * ReleaseActionParameters)} method should generate different release action
     * script code depending on whether the end user has enabled or disabled the
     * release detection option (in the Gripper program node). For more details,
     * see Javadoc for
     * {@link GripActionParameters#isGripDetectionEnabled()}.
     *
     * @param scriptCodeGenerator script generator that generates the script
     * code for determining if a release was detected. When the script code
     * needs to be generated, the method
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} will be called
     * (by AuboScope). The resulting script code will be embedded in a script
     * function and <b>must</b> return a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideGripperConfigurationPhase if this method is called
     * at the wrong time, i.e. outside the scope of the {@link
     * GripperContribution#configureGripper(GripperConfiguration,
     * GripperAPIProvider)} method.
     */
    virtual void registerReleaseDetectedCapability(
        std::function<void(ScriptWriterPtr scriptWriter)>
            scriptCodeGenerator) = 0;
};
} // namespace aubo_scope
} // namespace arcs

#endif
