#ifndef AUBO_SCOPE_GRIPPER_CONTRIBUTION_H
#define AUBO_SCOPE_GRIPPER_CONTRIBUTION_H

#include <aubo_caps/contribution/driver/gripper/gripper_api_provider.h>
#include <aubo_caps/contribution/driver/gripper/gripper_configuration.h>
#include <aubo_caps/contribution/driver/gripper/release_action_parameters.h>
#include <aubo_caps/contribution/driver/gripper/grip_action_parameters.h>

#include <aubo_caps/domain/system_api.h>
#include <aubo_caps/domain/script/script_writer.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(GripperContribution);

/**
 * <p>
 * This interface defines an API for adding functionality for gripping devices
 * to AuboScope.
 * </p>
 *
 * <p>
 * Implementing this interface will create a program node contribution, an
 * installation contribution and a toolbar contribution exposing gripper
 * functionality to the end user. Note that the toolbar contribution is not
 * supported on CB3 robots.
 * </p>
 *
 * <p>
 * The program node contribution allows the end user to program grip and release
 * actions. The configuration options available to the end user are based on the
 * properties and capabilities supported by the gripper. As a minimum, all
 * grippers must support "default" grip and release actions. All other
 * capabilities are optional to support and can be registered when the {@link
 * GripperContribution#configureGripper(GripperConfiguration,
 * GripperAPIProvider)} method is called.
 * </p>
 *
 * <p>
 * The installation node contribution can be used, if any custom input from the
 * end user is required to setup the gripper, e.g. specifying an IP address. The
 * corresponding generated UI will be accessible in the installation node. This
 * functionality is optional, since such configuration properties are specific
 * for each gripper. Access to create a custom configuration based on user
 * inputs is provided when the method
 * {@link
 * GripperContribution#configureInstallation(CustomUserInputConfiguration,
 * SystemConfiguration, TCPConfiguration, GripperAPIProvider)} is called. If no
 * custom user inputs, texts, etc. are defined, the installation node will not
 * be visible in AuboScope. Preamble script code (generated by {@link
 * #generatePreambleScript(ScriptWriter)}) will still be added to the robot
 * program.
 * </p>
 *
 * <p>
 * The toolbar contribution provides the end user quick access to perform grip
 * and release actions using "default" gripper action parameters.
 * </p>
 */
class GripperContribution
{
public:
    virtual ~GripperContribution() = default;

    /**
     * This method must return the title of the gripper contribution. The title
     * is displayed in: <ul> <li>The Structure section of the Program Tab for
     * this type of gripper program node (from which the Gripper program node
     * can be inserted)</li> <li>The program node screen for this gripper
     * contribution.</li> <li>The Program Tree as title for the program node for
     * this gripper contribution</li> <li>The left-hand side navigation of the
     * Installation Tab for this gripper contribution.</li> <li>The installation
     * node screen for this gripper contribution.</li> <li>The toolbar
     * contribution for this gripper. Note: The toolbar contribution is not
     * available on CB3 robots.</li>
     * </ul>
     *
     * <p>
     * The return value of this method cannot be <code>null</code> nor an empty
     * string.
     * </p>
     *
     * This method is called once upon startup.
     *
     * @param locale The current locale of AuboScope. Can be used for supporting
     * titles in several languages.
     * @return The title of this gripper contribution, not <code>null</code> nor
     * an empty string.
     */
    virtual std::string getTitle(/*Locale locale*/) = 0;

    /**
     * <p>
     * When this method is called, use the configuration parameter to register
     * or setup optional properties and capabilities of the gripper.
     * </p>
     *
     * <p>
     * The method is called in the Gripper Configuration phase after this
     * contribution has been registered when a new installation is loaded or
     * created.
     * </p>
     *
     * <b>Note:</b> If the gripper only supports basic "default" grip and
     * release actions, leave the implementation of this method empty.
     *
     * @param gripperConfiguration A configuration instance that can be used for
     * registering or setting up the properties and capabilities of the gripper.
     * @param apiProvider Provides access to functionality and services
     * available from within AuboScope which can be relevant for setting up the
     * gripper capabilities (e.g. the {@link SystemAPI} interface with
     * functionality for querying information about the robot).
     */
    virtual void configureGripper(GripperConfigurationPtr gripperConfiguration,
                                  GripperApiProviderPtr apiProvider) = 0;

    /**
     * <p>
     * When this method is called, the script code for performing a grip action
     *with the gripper must be generated.
     * </p>
     *
     * <p>
     * The parameters for the registered optional gripper capabilities
     *defined/configured by the end user are provided as input to the script
     *generation.
     * </p>
     *
     * <p>
     * The method is called in the following scenarios:
     *	<ul>
     * 	  <li> When the script code for a gripper program node configured with
     *a grip action should be generated as part of a robot program to be
     *executed </li> <li> When the end user tests a grip configuration of a
     *gripper program node </li> <li> When the gripper is operated using the
     *toolbar </li>
     * 	</ul>
     * </p>
     *
     * <b>NOTE:</b>
     *  <ul>
     *    <li> Applying a new total payload after an object has been gripped is
     *<b>not</b> the responsibility of the gripper, since this is automatically
     *handled by AuboScope. </li> <li> <p> When the end user has specified a new
     *payload (in the Gripper program node or toolbar), the user-defined payload
     *value will be applied by AuboScope immediately after the generated script
     *code for the grip action has finished executing. Hence the script code
     *should not finish earlier than when it is appropriate to apply the new
     *payload value (i.e. when the object has been gripped). This typically
     *means that the script code should wait a time period corresponding to
     *closing the gripper's "fingers" fully (if the gripper is vacuum operated
     *then long enough to achieve some level of vacuum).
     *      </p>
     *
     *      <p>
     *      If the grip detected feedback capability has been registered (using
     *      {@link
     *GripperFeedbackCapabilities#registerGripDetectedCapability(ScriptCodeGenerator)}),
     *different script code should be generated depending on whether the end
     *user has enabled or disabled the grip detection option (in the Gripper
     *program node). For more details, see Javadoc for
     *      {@link GripActionParameters#isGripDetectionEnabled()}.
     *      </p>
     *    </li>
     *  </ul>
     *
     * @param scriptWriter Use this script writer instance to generate the
     *script code for gripping
     * @param parameters The parameters for the gripper action
     *defined/configured by the end user
     */
    virtual void generateGripActionScript(
        ScriptWriterPtr scriptWriter, GripActionParametersPtr parameters) = 0;

    /**
     * <p>
     * When this method is called, the script code for performing a release
     *action with the gripper must be generated.
     * </p>
     *
     * <p>
     * The parameters for the registered optional gripper capabilities
     *defined/configured by the end user are provided as input to the script
     *generation.
     * </p>
     *
     * <p>
     * The method is called in the following scenarios:
     *	 <ul>
     * 	  <li> When the script code for a gripper program node configured with
     *a release action should be generated as part of a robot program to be
     *executed </li> <li> When the end user tests a release configuration of a
     *gripper program node </li> <li> When the gripper is operated using the
     *toolbar </li>
     * 	 </ul>
     * </p>
     *
     * <b>NOTE:</b>
     *  <ul>
     *    <li> Applying a new total payload after an object has been released is
     *<b>not</b> the responsibility of the gripper, since this is automatically
     *handled by AuboScope. </li> <li> <p> When the end user has specified a new
     *payload (in the Gripper program node or toolbar), the user-defined payload
     *value will be applied by AuboScope immediately after the generated script
     *code for the release action has finished executing. Hence the script code
     *should not finish earlier than when it is appropriate to apply the new
     *payload value (i.e. when the object has been released). This typically
     *means that the script code should wait a time period corresponding to
     *opening the gripper's "fingers" fully (if the gripper is vacuum operated
     *then long enough to achieve some level of vacuum).
     *      </p>
     *
     *      <p>
     *      If the release detected feedback capability has been registered
     *(using
     *      {@link
     *GripperFeedbackCapabilities#registerReleaseDetectedCapability(ScriptCodeGenerator)}),
     *different script code should be generated depending on whether the end
     *user has enabled or disabled the release detection option (in the Gripper
     *program node). For more details, see Javadoc for
     *      {@link ReleaseActionParameters#isReleaseDetectionEnabled()}.
     *      </p>
     *    </li>
     *  </ul>
     *
     * @param scriptWriter Use this script writer instance to generate the
     *script code for releasing the gripper
     * @param parameters The parameters for the gripper action
     *defined/configured by the end user
     */
    virtual void generateReleaseActionScript(
        ScriptWriterPtr scriptWriter,
        ReleaseActionParametersPtr parameters) = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
