#ifndef AUBO_SCOPE_RELEASE_ACTION_PARAMETERS_H
#define AUBO_SCOPE_RELEASE_ACTION_PARAMETERS_H

#include <aubo_caps/domain/script/script_writer.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(ReleaseActionParameters);
/**
 * <p>
 * This interface represents parameters for a release action defined/configured
 * by the end user.
 * </p>
 *
 * These parameters are used when generating script code for a release action
 * (see
 * {@link GripperContribution#generateReleaseActionScript(ScriptWriter,
 * ReleaseActionParameters)}).
 */
class ReleaseActionParameters
{
public:
    virtual ~ReleaseActionParameters() = default;

    /**
     * <p>
     * Get the selected width to be used for releasing the gripper.
     * </p>
     *
     * This is only valid, if the width capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerWidthCapability(double, double,
     * double, double, Length.Unit)}).
     *
     * @return The width to use for the release action
     * @throws UnsupportedOperationException if the gripper has not registered
     * the width capability
     */
    virtual double getWidth() = 0;

    /**
     * <p>
     * Get the selected speed to be used for releasing the gripper.
     * </p>
     *
     * This is only valid, if the speed capability has been registered (using
     * the method
     * {@link GripperCapabilities#registerSpeedCapability(double, double,
     * double, double, Speed.Unit)}).
     *
     * @return The speed to use for the release action
     * @throws UnsupportedOperationException if the gripper has not registered
     * the speed capability
     */
    virtual double getSpeed() = 0;

    /**
     * <p>
     * Use this method to determine whether the release detection option in the
     * Gripper program node is on or off.
     * </p>
     *
     * <p>
     * Note that this method will report that release detection is off, when
     * script code is to be generated due to the user testing a release
     * configuration of a gripper program or operating the gripper using the
     * toolbar.
     * </p>
     *
     * <p>
     * This method should only be called, if the release detected feedback
     * capability has been registered (using
     * {@link
     * GripperFeedbackCapabilities#registerReleaseDetectedCapability(ScriptCodeGenerator)}).
     * </p>
     *
     * <p>
     * The purpose of this method is to ensure that the appropriate script code
     * for performing a release action is generated when the end user has
     * specified a new payload value. The user-defined payload value will be
     * applied by AuboScope immediately after the generated release action
     * script code has finished executing. Depending on the enablement state of
     * release detection, the generated script should have the following
     * properties: <ul> <li> <i>Release detection off:</i> The script code
     * should not finish earlier than when it is appropriate to apply the new
     * payload value (i.e. when the object has been released). This typically
     * means that the script code should wait a time period corresponding to
     * opening the gripper's "fingers" fully (if the gripper is vacuum operated
     * then long enough to achieve some level of vacuum).
     *      </li>
     *      <li>
     * 	        <i>Release detection on:</i> The payload will only be set if the
     * release detection was successful, i.e. it will be applied immediately
     * after a release was detected. This means that it is not necessary for the
     * 	        script code to wait (since the built-in detection timeout
     * functionality will handle the waiting part).
     * 	     </li>
     *  </ul>
     * </p>
     *
     * @return <code>true</code> if the end user has enabled release detection,
     * and <code>false</code> if release detection is disabled.
     * @throws UnsupportedOperationException if the gripper has not registered
     * the release detected feedback capability
     */
    virtual bool isReleaseDetectionEnabled() = 0;
};
} // namespace aubo_scope
} // namespace arcs
#endif
