#ifndef AUBO_SCOPE_SCREW_DIRVER_CONFIGURATION_H
#define AUBO_SCOPE_SCREW_DIRVER_CONFIGURATION_H

#include <functional>
#include <aubo_caps/domain/script/script_writer.h>
#include <aubo_caps/contribution/driver/screw_driver/screw_driver_parameters.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(ProgramSelectionParameters);
/**
 * <p>
 * This interface provides parameters relevant for generating script code for
 * the Program Selection capability (registered using {@link
 * ScrewdriverCapabilities#registerProgramSelectionCapability(ScrewdriverProgramListProvider,
 * ScriptCodeGenerator)}).
 * </p>
 *
 * These parameters are passed when AuboScope calls the implementation of the
 * {@link ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method
 * responsible for the generating the script code.
 */
class ProgramSelectionParameters
{
public:
    virtual ~ProgramSelectionParameters() = default;

    /**
     * @return The selected screwdriver program to use for the operation.
     * @throws UnsupportedOperationException if the screwdriver has not
     * registered this capability
     */
    virtual ScrewdriverProgram getScrewdriverProgram() = 0;

    /**
     * @return The operation type selected by the end user.
     * @throws UnsupportedOperationException if the screwdriver has not
     * registered this capability
     */
    virtual ScrewdriverOperationType getScrewdriverOperationType() = 0;
};

ARCS_CLASS_FORWARD(PrepareToStartScrewdriverParameters);
/**
 * <p>
 * This interface provides parameters relevant for generating script code for
 * the Prepare To Start Screwdriver capability (registered using {@link
 * ScrewdriverCapabilities#registerPrepareToStartScrewdriverCapability(ScriptCodeGenerator)}).
 * </p>
 *
 * <p>
 * These parameters are passed when AuboScope calls the implementation of the
 * {@link ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method
 * responsible for the generating the script code.
 * </p>
 *
 * The interface is intentionally empty, but could have parameters in future
 * releases.
 */
class PrepareToStartScrewdriverParameters
{
public:
    virtual ~PrepareToStartScrewdriverParameters() = default;
};

ARCS_CLASS_FORWARD(FeedScrewParameters);
/**
 * <p>
 * This interface provides parameters relevant for generating script code for
 * the Feed Screw capability (registered using {@link
 * ScrewdriverCapabilities#registerFeedScrewCapability(ScriptCodeGenerator)}).
 * </p>
 *
 * <p>
 * These parameters are passed when AuboScope calls the implementation of the
 * {@link ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method
 * responsible for the generating the script code.
 * </p>
 *
 * The interface is intentionally empty, but could have parameters in future
 * releases.
 */
class FeedScrewParameters
{
public:
    virtual ~FeedScrewParameters() = default;
};

ARCS_CLASS_FORWARD(DriveScrewNotOKParameters);
/**
 * <p>
 * This interface provides parameters relevant for generating script code for
 * the Drive Screw Not OK feedback capability (registered using {@link
 * ScrewdriverFeedbackCapabilities#registerDriveScrewNotOKCapability(ScriptCodeGenerator)}).
 * </p>
 *
 * <p>
 * These parameters are passed when AuboScope calls the implementation of the
 * {@link ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method
 * responsible for the generating the script code.
 * </p>
 *
 * The interface is intentionally empty, but could have parameters in future
 * releases.
 */
class DriveScrewNotOKParameters
{
public:
    virtual ~DriveScrewNotOKParameters() = default;
};

ARCS_CLASS_FORWARD(DriveScrewOKParameters);
/**
 * <p>
 * This interface provides parameters relevant for generating script code for
 * the Drive Screw OK feedback capability (registered using {@link
 * ScrewdriverFeedbackCapabilities#registerDriveScrewOKCapability(ScriptCodeGenerator)}).
 * </p>
 *
 * <p>
 * These parameters are passed when AuboScope calls the implementation of the
 * {@link ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method
 * responsible for the generating the script code.
 * </p>
 *
 * The interface is intentionally empty, but could have parameters in future
 * releases.
 */
class DriveScrewOKParameters
{
public:
    virtual ~DriveScrewOKParameters() = default;
};

ARCS_CLASS_FORWARD(ScrewdriverReadyParameters);
/**
 * This interface provides the parameters relevant for generating script for the
 * Screwdriver Ready feedback operation capability (registered using {@link
 * ScrewdriverFeedbackCapabilities#registerScrewdriverReadyCapability(ScriptCodeGenerator)}).
 *
 * <p>
 * These parameters are passed when AuboScope calls the implementation of the
 * {@link ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method
 * responsible for the generating the script code.
 * </p>
 *
 * The interface is intentionally empty, but could have parameters in future
 * releases.
 */
class ScrewdriverReadyParameters
{
public:
    virtual ~ScrewdriverReadyParameters() = default;
};

ARCS_CLASS_FORWARD(ScrewdriverConfiguration);
/**
 * This interface provides access to register or setup various properties and
 * capabilities of a screwdriver. It is optional to register these capabilities
 * and properties.
 */
class ScrewdriverConfiguration
{
public:
    virtual ~ScrewdriverConfiguration() = default;

    /**
     * Add a screwdriver program to the list of programs.
     *
     * @param screwdriverProgram program to add, not <code>null</code>
     * @throws DuplicateScrewdriverProgramID if the ids of the screwdriver
     * programs (returned by the method {@link ScrewdriverProgram#getId()})
     * are not unique.
     * @throws NameOfScrewdriverProgramCannotBeNullOrEmpty if the display name
     * of a program (returned by the method {@link
     * ScrewdriverProgram#getDisplayName()}) is <code>null</code> or an empty
     * string.
     */
    virtual void addScrewdriverProgram(
        ScrewdriverProgramPtr screwdriverProgram) = 0;

    /**
     * Add a list of screwdriver programs to the list of programs.
     *
     * @param screwdriverProgramList list of programs to add, not
     * <code>null</code>
     * @throws DuplicateScrewdriverProgramID if the ids of the screwdriver
     * programs (returned by the method
     *                                       {@link ScrewdriverProgram#getId()})
     * are not unique.
     * @throws NameOfScrewdriverProgramCannotBeNullOrEmpty if the display name
     * of a program (returned by the method
     * 	                                                 {@link
     * ScrewdriverProgram#getDisplayName()}) is <code>null</code> or an empty
     * string.
     */
    virtual void addScrewdriverPrograms(
        std::vector<ScrewdriverProgramPtr> screwdriverProgramList) = 0;

    /**
     * <p>
     * Register a program selection capability for a screwdriver that supports
     * selecting between a set of screwdriver programs, typically defined on an
     * external control box for the screwdriver.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for selecting the specified screwdriver
     * program. The return value of the script code must be a boolean, i.e.
     * 'True' or 'False'. 'True' must be returned if selecting the program
     * succeeded and 'False' must be returned if the program selection failed.
     * </p>
     *
     * The generated script code is executed before the script code for starting
     * the screwdriver (could be from a different node), i.e the script code
     * generated by
     * {@link
     * ScrewdriverContribution#generateStartScrewdriverScript(ScriptWriter,
     * ScrewdriverParameters)}.
     *
     * @param programListProvider provider of the list of screwdriver programs,
     * not <code>null</code>
     * @param scriptCodeGenerator script code generator responsible for
     * generating the script code for selecting the specified screwdriver
     * program. When the script code needs to be generated, the
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method will be
     * called (by AuboScope). Return value for the generated script code must be
     * a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerProgramSelectionCapability(
        std::function<void(ScriptWriterPtr, ProgramSelectionParametersPtr)>
            scriptCodeGenerator) = 0;

    /**
     * Register an operation type capability for a screwdriver that
     * supports/requires the parameter for the screwdriving operation type (i.e.
     * if the operation is Tighten or Loosen).
     *
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerOperationTypeCapability() = 0;

    /**
     * <p>
     * Register a capability/requirement for a screwdriver that needs to
     * generate script code for preparing the screwdriver for a screwdriving
     * operation (before starting the screw driver).
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for preparing the screwdriver. The return
     * value of the script code must be a boolean, i.e. 'True' or 'False'.
     * 'True' must be returned if the preparation of the screwdriver was
     * successful and 'False' must be returned if the preparation failed.
     * </p>
     *
     * The generated script code is executed before the script code for starting
     * the screwdriver (could be from a different node), i.e the script code
     * generated by
     * {@link
     * ScrewdriverContribution#generateStartScrewdriverScript(ScriptWriter,
     * ScrewdriverParameters)}.
     *
     * @param scriptCodeGenerator script code generator responsible for
     * generating the script code for preparing the screwdriver. When the script
     * code needs to be generated, the
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method will be
     * called (by AuboScope). Return value for the generated script code must be
     * a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerPrepareToStartScrewdriverCapability(
        std::function<void(ScriptWriterPtr,
                           PrepareToStartScrewdriverParametersPtr)>
            scriptCodeGenerator) = 0;

    /**
     * <p>
     * Register a capability for a screwdriver that supports automatic screw
     * feeding.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for feeding a screw to the screwdriver. The
     * return value of the script code must be a boolean, i.e. 'True' or
     * 'False'. 'True' must be returned if the screw feed operation was
     * successful and 'False' must be returned if the operation failed.
     * </p>
     *
     * The generated script code is executed before the script code for
     * preparing the screwdriver, i.e the script code generated by {@link
     * #registerPrepareToStartScrewdriverCapability(ScriptCodeGenerator)}.
     *
     * @param scriptCodeGenerator script code generator responsible for
     * generating the script code for feeding a screw to the screwdriver. When
     * the script code needs to be generated, the
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method will be
     * called (by AuboScope). Return value for the generated script code must be
     * a boolean, i.e. 'True' or 'False'
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerFeedScrewCapability(
        std::function<void(ScriptWriterPtr, FeedScrewParametersPtr)>
            scriptCodeGenerator) = 0;

    /**
     * <p>
     * Register a feedback capability for a screwdriver that is capable of
     * providing information about whether or not an ongoing screwdriving
     * operation (driving the screw) ended successfully (ended in OK-state).
     * </p>
     *
     * <p>
     * <b>Note:</b> This capability must only be used to provide feedback about
     * the success of the screwing operation. It should not be used to provide
     * feedback about failures. To support feedback about when the operation
     * fails, register the Drive Screw Not Ok feedback capability the using the
     * {@link #registerDriveScrewNotOKCapability(ScriptCodeGenerator)} method.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for determining if the screwing operation
     * succeeded. The return value of the script code must be a boolean, i.e.
     * 'True' or 'False'. 'True' must be returned if the screwing operation was
     * successful, otherwise 'False' must be returned.
     * </p>
     *
     * <p>
     * <b>Note:</b> A return value of 'False' must not be used to indicate that
     * the operation failed, but simply that a success was not (yet) detected.
     * If the screwdriver can provide feedback on whether or not the screwing
     * operation failed, the Drive Screw Not Ok feedback capability must be
     * used.
     * </p>
     *
     * <b>Example:</b> <br>
     * The digital tool input 0 will go high if the operation ended in OK-state.
     * The script code is: <pre> return get_tool_digital_in(0)
     * </pre>
     *
     * So the implementation of {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} could be: <pre>
     * scriptWriter.appendLine("return get_tool_digital_in(0)");
     * </pre>
     *
     * @param scriptCodeGenerator script code generator responsible for
     * generating the script code for determining if the screwing operation
     * succeeded. When the script code needs to be generated, the
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method will be
     *                            called (by AuboScope). Return value for the
     * generated script code must be a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerDriveScrewOKCapability(
        std::function<void(ScriptWriterPtr, DriveScrewOKParametersPtr)>
            scriptCodeGenerator) = 0;

    /**
     * <p>
     * Register a feedback capability for a screwdriver that is capable of
     * providing information about whether or not an ongoing screwdriving
     * operation (driving the screw) failed, i.e. ended in Not Ok (NOK) state.
     * </p>
     *
     * <p>
     * <b>Note:</b> This capability must only be used to provide feedback about
     * the failure of the screwing operation. It should not be used to provide
     * feedback about successes. To support feedback about when the operation
     * succeeds, register the Drive Screw Ok feedback capability the using the
     * {@link #registerDriveScrewOKCapability(ScriptCodeGenerator)} method.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for determining if the screwing operation
     * ended in failure. The return value of the script code must be a boolean,
     * i.e. 'True' or 'False'. 'True' must be returned if the screwing operation
     * failed, otherwise 'False' must be returned.
     * </p>
     *
     * <p>
     * <b>Note:</b> A return value of 'False' must not be used to indicate that
     * the operation succeeded, but simply that a failure was not (yet)
     * detected. If the screwdriver can provide feedback on whether or not the
     * screwing operation succeeded, the Drive Screw Ok feedback capability
     * should be used.
     * </p>
     *
     * <b>Example:</b> <br>
     * The digital tool input 1 will go high if the operation ended in NOT
     * OK-state. The script code is: <pre> return get_tool_digital_in(1)
     * </pre>
     * So the implementation of {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} could be: <pre>
     *    scriptWriter.appendLine("return get_tool_digital_in(1)");
     * </pre>
     *
     *
     * @param scriptCodeGenerator script code generator responsible for
     * generating the script code for determining if the screwing operation
     * failed. When the script code needs to be generated, the {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method will be
     *                            called (by AuboScope). Return value for the
     * generated script code must be a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerDriveScrewNotOKCapability(
        std::function<void(ScriptWriterPtr, DriveScrewNotOKParametersPtr)>
            scriptCodeGenerator) = 0;

    /**
     * <p>
     * Register a feedback capability for a screwdriver that is capable of
     * providing information about whether or not the screwdriver device is
     * ready to operate (receive commands). This capability is used (by
     * AuboScope) in a program to ensure that the screwdriver is ready before
     * sending commands to it.
     * </p>
     *
     * <p>
     * The provided implementation of the {@link ScriptCodeGenerator} interface
     * must generate the script code for determining if the screwdriver is
     * ready. The return value of the script code must be a boolean, i.e. 'True'
     * or 'False'. 'True' must be returned if the screwdriver is ready,
     * otherwise 'False' must be returned.
     * </p>
     *
     * The generated script code is executed before the script code for the
     * program selected capability (if that capability has been registered).
     *
     * @param scriptCodeGenerator script code generator responsible for
     * generating the script code for determining if the screwdriver is ready.
     * When the script code needs to be generated, the
     *                            {@link
     * ScriptCodeGenerator#generateScript(ScriptWriter, Object)} method will be
     *                            called (by AuboScope). Return value for the
     * generated script code must be a boolean, i.e. 'True' or 'False'.
     * @throws CapabilityAlreadyRegistered if this capability has already been
     * registered
     * @throws CalledOutsideScrewdriverConfigurationPhase if this method is
     * called at the wrong time, i.e. outside the scope of the {@link
     * ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
     * ScrewdriverApiProvider)} method.
     */
    virtual void registerScrewdriverReadyCapability(
        std::function<void(ScriptWriterPtr, ScrewdriverReadyParametersPtr)>
            scriptCodeGenerator) = 0;
};
} // namespace aubo_scope
} // namespace arcs

#endif
