#ifndef AUBO_SCOPE_SCREW_DIRVER_CONTRIBUTION_H
#define AUBO_SCOPE_SCREW_DIRVER_CONTRIBUTION_H

#include <aubo_caps/domain/system_api.h>
#include <aubo_caps/domain/script/script_writer.h>
#include <aubo_caps/contribution/driver/screw_driver/screw_driver_configuration.h>
#include <aubo_caps/contribution/driver/screw_driver/screw_driver_parameters.h>
#include <aubo_caps/contribution/driver/screw_driver/screw_driver_api_provider.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ScrewdriverContribution);

/**
 * <p>
 * This interface defines an API for adding functionality for screwdrivers to
 * AuboScope.
 * </p>
 *
 * <p>
 * Implementing this interface will "integrate" (hook) this screwdriving
 * contribution into the builtin (native) AuboScope Screwdriving program node
 * and Screwdriving installation screen. This will expose the screwdriving
 * functionality supported by the screwdriver to the end user.
 * </p>
 *
 * <p>
 * <b>Note:</b> Both the builtin (native) AuboScope Screwdriving program node
 * and Screwdriver contributions are not supported on CB3 robots. If a AuboCap
 * registers a Screwdriver contribution on a CB3 robot, the AuboCap will fail to
 * be activated.
 * </p>
 *
 * <p>
 * The Screwdriving program node allows the end user to program screwdriving
 * operations. The configuration options available to the end user are based on
 * the properties and capabilities supported by the screwdriver. As a minimum,
 * all screwdrivers must support "default" screw (start screwdriver) and stop
 * (stop screwdriver) operations. All other capabilities are optional to support
 * and can be registered when the method
 * {@link ScrewdriverContribution#configureScrewdriver(ScrewdriverConfiguration,
 * ScrewdriverApiProvider)} is called.
 * </p>
 *
 * <p>
 * The integration in the Screwdriving installation screen allows the user to
 * select the screwdriver for programming screwing operations (using the
 * Screwdriving program node). It also offers the possibility to define any
 * custom inputs from the end user that are required to setup the screwdriver,
 * e.g. specifying an IP address. This functionality is optional, since such
 * configuration properties are specific for each screwdriver device. The
 * corresponding generated custom UI will be accessible in the screen when the
 * screwdriver is selected (by the end user) and replace the default
 * Screwdriving installation UI. Access to create a custom configuration based
 * on user inputs is provided when the method
 * {@link
 * ScrewdriverContribution#configureInstallation(CustomUserInputConfiguration,
 * SystemConfiguration, TCPConfiguration, ScrewdriverApiProvider)} is called.
 * </p>
 */
class ScrewdriverContribution
{
public:
    virtual ~ScrewdriverContribution() = default;

    /**
     * This method must return the title of the screwdriver contribution. The
     * title is displayed in: <ul> <li>The builtin (native) AuboScope
     * Screwdriving installation screen at the top where the screwdriver is
     * selected
     *     </li>
     *     <li>The screen for the builtin (native) AuboScope Screwdriving
     * program node (if this screwdriver has been selected in the Screwdriving
     * installation screen)
     * 	   </li>
     * </ul>
     *
     * This method is called once upon startup.
     *
     * @param locale the current locale of AuboScope. Can be used for supporting
     * titles in several languages.
     * @return the title of this screw driver contribution, not
     * <code>null</code> nor an empty string.
     */
    virtual std::string getTitle(/*Locale locale*/) = 0;

    /**
     * <p>
     * When this method is called, use the configuration parameter to register
     * or setup optional properties and capabilities of the screwdriver.
     * </p>
     *
     * <p>
     * The method is called in the Screwdriver Configuration phase after this
     * contribution has been registered when a new installation is loaded or
     * created.
     * </p>
     *
     * <b>Note:</b> If the screwdriver only supports basic "default" screw (see
     * {@link #generateStartScrewdriverScript(ScriptWriter,
     * ScrewdriverParameters)}) and stop (see
     * {@link #generateStopScrewdriverScript(ScriptWriter,
     * ScrewdriverParameters)}) operations, leave the implementation of this
     * method empty.
     *
     * @param screwdriverConfiguration a configuration instance that can be used
     * for registering or setting up the properties and capabilities of the
     * screwdriver.
     * @param apiProvider provides access to functionality and services
     * available from within AuboScope which can be relevant for setting up the
     * screwdriver capabilities (e.g. the {@link SystemAPI} interface with
     * functionality for querying information about the robot).
     */
    virtual void configureScrewdriver(
        ScrewdriverConfiguration screwdriverConfiguration,
        ScrewdriverAPIProviderPtr apiProvider);

    /**
     * <p>
     * When this method is called, the script code for driving the screw must be
     * generated.
     * </p>
     *
     * The relevant parameters for the registered optional screwdriver
     * capabilities/requirements defined/configured by the end user are provided
     * as input to the script code generation.
     *
     * @param scriptWriter use this script writer instance to generate the
     * script code for driving the screw
     * @param parameters the parameters for the screwdriving operation
     * defined/configured by the end user
     */
    virtual void generateStartScrewdriverScript(
        ScriptWriterPtr scriptWriter, ScrewdriverParametersPtr parameters) = 0;

    /**
     * <p>
     * When this method is called, the script code for stopping the screw driver
     * must be generated (i.e. the screwdriver must stop the current
     * screwdriving operation).
     * </p>
     *
     * The relevant parameters for the registered optional screwdriver
     * capabilities/requirements defined/configured by the end user are provided
     * as input to the script code generation.
     *
     * @param scriptWriter use this script writer instance to generate the
     * script code for stopping the screwdriver
     * @param parameters the parameters for the screwdriving operation
     * defined/configured by the end user
     */
    virtual void generateStopScrewdriverScript(
        ScriptWriterPtr scriptWriter, ScrewdriverParametersPtr parameters) = 0;
    ;
};
} // namespace aubo_scope
} // namespace arcs

#endif
