#ifndef AUBO_SCOPE_SCREW_DIRVER_PARAMETERS_H
#define AUBO_SCOPE_SCREW_DIRVER_PARAMETERS_H

#include <string>
#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(ScrewdriverProgram);
/**
 * This abstract class represents a screwdriver program, typically defined on an
 * external control box for the screwdriver.
 */
class ScrewdriverProgram
{
public:
    virtual ~ScrewdriverProgram() = 0;

    /**
     * <p>
     * This method must return an identifier to uniquely identify this
     * screwdriver program. The identifier will be used for persistence and
     * resolving the correct screwdriver program. Therefore the identifier must
     * remain constant over time and versions of the screwdriver AuboCap.
     * </p>
     *
     * This method must be implemented.
     *
     * @return a unique identifier string which will remain constant over time
     * and versions of the screwdriver AuboCap.
     */
    virtual std::string getId() = 0;

    /**
     * <p>
     * This method should return the name of the screwdriver program. The name
     * will be displayed in the UI, e.g. in a combo box. The name returned is
     * allowed to change over versions of the driver AuboCap as well as be
     * different depending on the current language selected in AuboScope, in
     * case translations are supported.
     * </p>
     *
     * This method must be implemented.
     *
     * @return the name of the screwdriver program to display in the UI.
     */
    virtual std::string getDisplayName() = 0;
};

/**
 * This interface represents the type of screwing operation performed by the
 * screwdriver.
 */
enum ScrewdriverOperationType
{

    /**
     * The screwdriver operation will follow the screw in a tightening direction
     * (in).
     */
    TIGHTEN,

    /**
     * The screwdriver operation will follow the screw in a loosening direction
     * (out).
     */
    LOOSEN
};

ARCS_CLASS_FORWARD(ScrewdriverParameters);

/**
 * This interface represents parameters for a screwdriver operation
 * defined/configured by the end user.
 */
class ScrewdriverParameters
{
public:
    virtual ~ScrewdriverParameters() = default;

    /**
     * <p>
     * Get the selected screwdriver program to be used for the screwdriver
     * operation.
     * </p>
     *
     * This is only valid, if the program selection capability has been
     * registered (using the method
     * {@link
     * ScrewdriverCapabilities#registerProgramSelectionCapability(ScrewdriverProgramListProvider,
     * ScriptCodeGenerator)}).
     *
     * @return The selected screwdriver program to use for the operation.
     * @throws UnsupportedOperationException if the screwdriver has not
     * registered the program selection capability
     */
    virtual ScrewdriverProgram getScrewdriverProgram() = 0;

    /**
     * <p>
     * Get the selected screwdriver operation type.
     * </p>
     *
     * This is only valid, if the operation type capability has been registered
     * (using the method
     * {@link ScrewdriverCapabilities#registerOperationTypeCapability()}).
     *
     * @return The operation type selected by the user.
     * @throws UnsupportedOperationException if the screwdriver has not
     * registered the operation type capability
     */
    virtual ScrewdriverOperationType getScrewdriverOperationType() = 0;
    ;
};
} // namespace aubo_scope
} // namespace arcs

#endif
