#ifndef AUBO_SCOPE_INSTALLATION_SERVICE_H
#define AUBO_SCOPE_INSTALLATION_SERVICE_H

#include <string>
#include <aubo_caps/domain/data/data_model.h>
#include <aubo_caps/contribution/view_api_provider.h>
#include <aubo_caps/contribution/installation/installation_node_view.h>
#include <aubo_caps/contribution/installation/installation_node_contribution.h>
#include <aubo_caps/contribution/installation/installation_creation_context.h>
#include <aubo_caps/contribution/installation/installation_contribution_configuration.h>
#include <aubo_caps/contribution/installation/installation_api_provider.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(InstallationNodeService);

/**
 * Defines an API required for defining and adding to AuboScope an installation
 * node and corresponding screen where the user interface is Swing-based.
 */
class InstallationNodeService
{
public:
    virtual ~InstallationNodeService() = default;

    /**
     * @param locale the current locale of AuboScope. Can be used for supporting
     * titles in several languages.
     * @return The text displayed for this installation contribution in the
     * left-hand side navigation of the Installation Tab as well as the title of
     * the corresponding installation node screen. Called once at start up.
     */
    virtual std::string getTitle() = 0;

    /**
     * @param locale the current locale of AuboScope. Can be used for supporting
     * titles in several languages.
     * @return The text displayed for this installation contribution in the
     * left-hand side navigation of the Installation Tab as well as the title of
     * the corresponding installation node screen. Called once at start up.
     */
    virtual std::string getIcon() = 0;

    /**
     * This method is called once after this service is registered. Modify the
     * configuration parameter to configure your contribution/ The configuration
     * object will already have default values for its properties matching most
     * use cases. <p> The values of the ContributionConfiguration object will be
     * read once immediately after this method call. Changing values at a later
     * stage will have no effect, so do not store a reference to the
     * configuration object.
     * </p>
     * If the default values are appropriate, leave this method empty.
     *
     * @param configuration a modifiable ContributionConfiguration with default
     * values
     */
    virtual void configureContribution(
        InstallationContributionConfigurationPtr configuration) = 0;

    /**
     * Creates a new View instance which implements the UI for your installation
     * node screen. Called once when a new installation is loaded or created.
     *
     * @param api_provider Provides access to functionality and services
     * available from within AuboScope related to user interface and end user
     * interaction
     * @return the view
     */
    virtual InstallationNodeViewPtr createView(
        ViewApiProviderPtr api_provider) = 0;

    /**
     * <p>
     * Creates a new installation node instance.
     * </p>
     * <p>
     * The returned node must use the supplied data model object to retrieve and
     * store the data contained in it. All modifications to the supplied data
     * model from the installation node constructor are ignored when an existing
     * installation is loaded. The data model object is shared between all
     * installation nodes contributed by the same AuboCap.
     * </p>
     *
     * @param api_provider Provides access to functionality and services
     * available from within AuboScope relevant for the installation node
     * @param view the View created by {@link #createView(ViewAPIProvider)}
     * @param model object where all configuration data of the new installation
     * node instance is to be stored in and retrieved from
     * @param context the context in which this node is created
     * @return the newly constructed installation node contribution instance
     */
    virtual InstallationNodeContributionPtr createInstallationNode(
        InstallationApiProviderPtr api_provider, InstallationNodeViewPtr view,
        DataModelPtr model, InstallationCreationContextPtr context) = 0;
};

} // namespace aubo_scope
} // namespace arcs

#endif
