#ifndef AUBO_SCOPE_CONTRIBUTION_CONFIGURATION_H
#define AUBO_SCOPE_CONTRIBUTION_CONFIGURATION_H

#include <any>
#include <vector>
#include <functional>

#include <aubo_caps/domain/program/structure/tree_node.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(ProgramContributionConfiguration);

/**
 * Provides access to configure the properties of an installation node
 * contribution/
 */
class ProgramContributionConfiguration
{
public:
    ProgramContributionConfiguration(ProgramContributionConfiguration &f);
    ProgramContributionConfiguration(ProgramContributionConfiguration &&f);
    virtual ~ProgramContributionConfiguration();

    /**
     * If set to <code>true</code> it is possible for the program node to have
     * child nodes.
     *
     * <p>
     * The default value is <code>false</code>.
     * </p>
     *
     * @param childrenAllowed determines whether this node can have child nodes
     */
    void setChildrenAllowed(bool childrenAllowed);
    void setChildrenAllowedOnly(const std::vector<std::string> &childrenList);

    /**
     * If set to <code>false</code> this program node will not appear in the
     * Program Tab in AuboScope and can not be inserted in the program tree by
     * the end user. It can only be inserted programmatically using the
     * {@link ProgramNodeFactory}. This
     * also means, it can not be inserted using copy/paste of that node
     * directly. It can however be inserted by copy/paste of an ancestor of such
     * a node.
     *
     * <p>
     * The default value is <code>true</code>.
     * </p>
     *
     * @param userInsertable determines whether the end user can insert this
     * node
     */
    void setUserInsertable(bool userInsertable);

    /**
     * If set to <code>true</code> it is not possible for the end user to create
     * new program nodes of this type. Loading of existing programs will however
     * still be supported for this program node.
     *
     * <p>
     * The default value is <code>false</code>.
     * </p>
     *
     * @param deprecated determines whether this node is deprecated
     */
    void setDeprecated(bool deprecated);

    /**
     * Set the display order id for this type of program node contribution. This
     * makes it possible to specify how all the program node contributions from
     * a AuboCap should be ordered/sorted in AuboScope.
     *
     * <p>
     * All program node contributions from a AuboCap will be grouped together
     * and sorted (in ascending order) according to their display order id.
     * </p>
     *
     * <p>
     * The default value is 0.0. If the display order id for all program node
     * contributions in a AuboCap have the default value, the standard built-in
     * sorting in AuboScope will be used.
     * </p>
     *
     * @param displayOrderId Determines the display order of the AuboCap program
     * node type
     */
    void setDisplayOrderId(double displayOrderId);

    /**
     * <p>
     * Configure whether or not the end user is allowed to set a breakpoint on
     * or single step the AuboCap program node.
     * </p>
     *
     * <p>
     * The default value is <code>true</code>.
     * </p>
     *
     * @param isBreakpointAllowed <code>true</code> if it should be allowed to
     * set a breakpoint on or single step this AuboCap program node. If it
     * should be disallowed, the value must be <code>false</code>.
     */
    void setAllowBreakpointOnNode(bool isBreakpointAllowed);

    /**
     * <p>
     * Configure whether or not the end user is allowed to set a breakpoint on
     * or single step any of the child nodes in the sub-tree under the AuboCap
     * program node. Note that it will only be possible to set a breakpoint on a
     * child node if the child node itself supports breakpoints.
     * </p>
     *
     * <p>
     * The default value is <code>false</code>.
     * </p>
     *
     * @param isBreakpointOnChildNodesAllowed <code>true</code> if it should be
     * allowed to set a breakpoint on or single step any of the child nodes in
     * the sub-tree under this AuboCap program node. If it should be disallowed,
     * the value must be <code>false</code>.
     */
    void setAllowBreakpointOnChildNodesInSubtree(
        bool isBreakpointOnChildNodesAllowed);

    /**
     * <p>
     * Configure whether or not the end user can start the program from the
     * AuboCap program node when it is selected in the program tree.
     * </p>
     *
     * <p>
     * The default value is <code>true</code>.
     * </p>
     *
     * @param isStartFromNodeAllowed <code>true</code> if it should be allowed
     * to start the program directly from this AuboCap program node. If it
     * should be disallowed, the value must be <code>false</code>.
     *
     */
    void setAllowStartFromNode(bool isStartFromNodeAllowed);

    /**
     * <p>
     * Configure whether or not the end user can start the program directly from
     * any selected child node in the sub-tree under the AuboCap program node.
     * Note that it will only be possible to start from a selected child node if
     * the child node itself supports starting from the child node.
     * </p>
     *
     * <p>
     * The default value is <code>false</code>.
     * </p>
     *
     * @param isStartFromChildNodesAllowed <code>true</code> if it should be
     * allowed to start the program directly from any child program node. If it
     * should be disallowed, the value must be <code>false</code>..
     */
    void setAllowStartFromChildNodesInSubtree(
        bool isStartFromChildNodesAllowed);

    /**
     * Configure parent node of this node. End user cannot insert this node to
     *
     * The default value is <code>""</code>.
     * @param parentNodeName
     */
    void setAllowInsertOnlyIntoNodes(
        const std::vector<std::string> &parentNodeNames);

private:
    ProgramContributionConfiguration();
    friend class DataSwitch;
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_CONTRIBUTION_CONFIGURATION_H
