#ifndef AUBO_SCOPE_PROGRAM_NODE_CONTRIBUTION_H
#define AUBO_SCOPE_PROGRAM_NODE_CONTRIBUTION_H

#include <string>
#include <aubo_caps/contribution/program/program_creation_context.h>
#include <aubo_caps/domain/script/script_writer.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ProgramNodeContribution);

/**
 * Defines an API required for specifying a AuboCap Program Node within
 * AuboScope.
 */
class ProgramNodeContribution
{
public:
    virtual ~ProgramNodeContribution() = default;

    /**
     * Called when this node is selected in the program tree.
     */
    virtual void openView() = 0;

    /**
     * Called when this node is unselected in the program tree or when
     * navigating to another view.
     */
    virtual void closeView() = 0;

    /**
     *
     * @return The text displayed in the Program Tree for the program node.
     * @throws IllegalStateException if the data model or child program
     * sub-tree is modified
     */
    virtual std::string getTitle() = 0;

    /**
     * <p>
     * Should return <code>true</code> if this program node is complete, and
     * <code>false</code> otherwise.
     * </p>
     *
     * <p>
     * Undefined program nodes will be marked with yellow in the program
     * tree. The user cannot start the program as long as it contains one or
     * more undefined program nodes.
     * </p>
     *
     * If this program node stores an unresolvable entity (e.g. {@link
     * variable/PersistedVariable} or {@link tcp/TCP}) in its DataModel, the
     * program node will be treated as undefined.
     *
     * @return <code>true</code> if this program node is complete and otherwise
     * <code>false</code>.
     * @throws IllegalStateException if the data model or child program
     * sub-tree is modified.
     */
    virtual bool isDefined() = 0;

    /**
     * Defines the script code to be generated by this program node within
     * the program.
     *
     * @param writer serves to add script commands that define the behaviour
     * of this program node.
     */
    virtual void generateScript(ScriptWriterPtr script_writer) = 0;

    /**
     * @brief post
     * @param pattern
     * @param body
     * @return
     */
    virtual std::string postAction(const char *pattern, const std::string &body)
    {
    }

    /**
     * @brief GET
     * @param pattern
     * @param body
     * @return
     */
    virtual std::string getAction(const char *pattern, const std::string &body)
    {
    }
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_PROGRAM_NODE_CONTRIBUTION_H
