#ifndef AUBO_SCOPE_PROGRAM_NODE_SERVICE_H
#define AUBO_SCOPE_PROGRAM_NODE_SERVICE_H

#include <aubo_caps/contribution/view_api_provider.h>
#include <aubo_caps/contribution/program/program_node_view.h>
#include <aubo_caps/contribution/program/program_creation_context.h>
#include <aubo_caps/contribution/program/program_api_provider.h>
#include <aubo_caps/contribution/program/program_contribution_configuration.h>
#include <aubo_caps/domain/data/data_model.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ProgramNodeService);

/**
 * Defines an API required for adding a new type of program node to AuboScope.
 */
class ProgramNodeService
{
public:
    virtual ~ProgramNodeService() = default;

    /**
     * Get the unique identifier for this kind of program node. The identifier
     * will be used when storing programs that use the program node.
     *
     * @return the unique identifier.
     */
    virtual std::string getId() = 0;

    /**
     *
     * @return The text displayed in the Structure Tab for program nodes created
     * by this factory.
     */
    virtual std::string getTitle() = 0;

    /**
     *
     * @return The text displayed in the Structure Tab for program nodes created
     * by this factory.
     */
    virtual std::string getIcon() = 0;

    /**
     * configureContribution
     * @param configuration
     */
    virtual void configureContribution(
        ProgramContributionConfigurationPtr configuration) = 0;

    /**
     * Creates a new View instance which implements the UI for your program node
     * screen. Called once when a new program is loaded or created.
     *
     * @param apiProvider Provides access to functionality and services
     * available from within AuboScope related to user interface and end user
     * interaction
     * @return the view
     */
    virtual ProgramNodeViewPtr createView(ViewApiProviderPtr api_provider) = 0;

    /**
     *
     * <p>
     * Creates a new program node contribution instance.
     * </p>
     *
     * <p>
     * The returned node must use the supplied data model object to retrieve and
     * store the data contained in it. All modifications to the supplied data
     * model from the program node constructor are ignored when an existing
     * program node is loaded or copy/pasted. Every change to the model object
     * is registered as a separate undo/redo event in the program tree.
     * </p>
     *
     * @param api the AuboCap API object with access to domain data
     * @param model object where all configuration data of the new program node
     * instance is to be stored in and retrieved from
     * @return the newly constructed program node contribution instance
     */
    virtual ProgramNodeContributionPtr createNode(
        ProgramApiProviderPtr api_provider, ProgramNodeViewPtr view,
        DataModelPtr model, ProgramCreationContextPtr context) = 0;
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_PROGRAM_NODE_SERVICE_H
