#ifndef AUBO_SCOPE_APPLICATION_API_H
#define AUBO_SCOPE_APPLICATION_API_H

#include <aubo_caps/domain/io/io_model.h>
#include <aubo_caps/domain/device/device_manager.h>
#include <aubo_caps/domain/feature/feature_model.h>
#include <aubo_caps/domain/payload/payload_model.h>
#include <aubo_caps/domain/tcp/tcp_model.h>
#include <aubo_caps/domain/variable/variable_model.h>
#include <aubo_caps/domain/value/value_factory_provider.h>

namespace arcs {

namespace aubo_sdk {
ARCS_CLASS_FORWARD(RobotProxy);
}

namespace aubo_scope {
ARCS_CLASS_FORWARD(ApplicationApi);
ARCS_CLASS_FORWARD(InstallationNodeContribution);

/**
 * Provides access to functionality and services which are relevant for various
 * different applications and contexts. This includes what is available from
 * within AuboScope, as well as creating additional model elements to be used
 * within AuboScope.
 */
class ARCS_ABI_EXPORT ApplicationApi
{
public:
    ApplicationApi(ApplicationApi &f);
    ApplicationApi(ApplicationApi &&f);
    virtual ~ApplicationApi();

    /**
     * @return An interface for working with the inputs and outputs of the robot
     */
    IoModelPtr getIoModel();

    /**
     * @return An interface for working with the features of the current
     * installation
     */
    FeatureModelPtr getFeatureModel();

    /**
     * @return An interface for accessing the TCPs of the current installation
     */
    TcpModelPtr getTcpModel();

    /**
     *
     * Gets an interface which provides access the payloads in the current
     * AuboScope installation.
     *
     * @return An interface for accessing the payloads in the current AuboScope
     * installation
     */
    PayloadModelPtr getPayloadModel();

    /**
     * @return An interface for working with variables
     */
    VariableModelPtr getVariableModel();

    /**
     * @return An interface for accessing various factories capable of creating
     * value objects.
     */
    ValueFactoryPtr getValueFactory();

    //    /**
    //	 * @return An interface with access to various queryable-only system
    // resources
    //	 */
    //	ResourceModel getResourceModel();

    /**
     * This method can be used to get a specific {@link
     * InstallationNodeContribution} instance
     *
     * @param installationType The class of the installation node
     * contribution to return
     * @param <T> The generic for specifying the {@link
     * InstallationNodeContribution}
     * @return The installation node instance
     */
    template <typename T>
    std::shared_ptr<T> getInstallationNode()
    {
        return std::dynamic_pointer_cast<T>(
            getInstallationNode(typeid(T).name()));
    }
    InstallationNodeContributionPtr getInstallationNode(
        const char *typeidOfClass);

    /**
     * This method can be used to get a specific {@link DeviceManager}
     * instance
     *
     * @param deviceManagerClass The class for the device manager to return
     * @param <T> The generic type of class to return
     * @return An interface for registering and working with a specific
     * device
     */
    DeviceManagerPtr getDeviceManager();

    /**
     * @brief getRobotProxy
     * @return
     */
    aubo_sdk::RobotProxyPtr getRobotProxy();

protected:
    ApplicationApi();

private:
    friend class DataSwitch;
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_APPLICATION_API_H
