#ifndef AUBO_SCOPE_DATA_MODEL_H
#define AUBO_SCOPE_DATA_MODEL_H

#include <stdint.h>
#include <string>
#include <vector>
#include <set>

#include <aubo_caps/domain/feature/feature.h>
#include <aubo_caps/domain/variable/variable.h>
#include <aubo_caps/domain/tcp/tcp.h>
#include <aubo_caps/domain/io/io.h>
#include <aubo_caps/domain/waypoint/waypoint.h>
#include <aubo_caps/domain/payload/payload.h>
#include <aubo_caps/domain/value/expression/expression.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(DataModel);

/**
 * <p>
 * This interface is used for storing and managing data that represents the
 * current configuration of, e.g. a
 * {@link ProgramNodeContribution} or {@link InstallationNodeContribution}.
 * Methods for adding, removing, retrieving and changing values in a dictionary
 * are provided.
 * </p>
 *
 * <p>
 * Setting a new value for a key already in use, will overwrite the current
 * value with the new value. This happens regardless of the type of value (e.g.
 * storing the value <code>true</code> under the key <i>myBool</i>
 * and afterwards storing an <code>Angle</code> object under the same key will
 * overwrite the value <code>true</code> with the provided <code>Angle</code>
 * object).
 * </p>
 *
 * <p>
 * A auboCap installation screen has an underlying <code>DataModel</code>
 * object. That object is saved and loaded along with each AuboScope
 * installation.
 * </p>
 *
 * <p>
 * Similarly, each contributed program node instance has an underlying
 * <code>DataModel</code> object. That object is saved and loaded along with the
 * program where the node occurs. Undo/redo actions are supported for all
 * modifications to the <code>DataModel</code> object in HTML-based program node
 * contributions. Swing-based aubo_studio plugins must use the {@link
 * UndoRedoManager} to record the changes on the undo/redo stack.
 * </p>
 *
 * <p>
 * When retrieving an object, both key and object type must match what was
 * previously stored. This means that if a
 * <code>TCP</code> object was stored using the key <i>myAngle</i>, then
 * attempting to retrieve it using
 * {@link #get(std::string key, Angle default_value)} with the key
 * <i>myAngle</i> will not return the stored value, since the types do not
 * match. Instead the provided <code>default_value</code> will be returned.
 * </p>
 *
 */
class ARCS_ABI_EXPORT DataModel
{
public:
    DataModel(DataModel &f);
    DataModel(DataModel &&f);
    ~DataModel();

    /**
     * Assign a <code>bool</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, bool value);

    /**
     * Get the <code>bool</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    bool get(const std::string &key, bool default_value) const;

    /**
     * Assign an <code>int</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, int value);

    /**
     * Get the <code>int</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    int get(const std::string &key, int default_value) const;

    /**
     * Assign a <code>long</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, long value);
    void set(const std::string &key, uint64_t value);

    /**
     * Get the <code>long</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    long get(const std::string &key, long default_value) const;
    uint64_t get(const std::string &key, uint64_t default_value) const;

    /**
     * Assign a <code>float</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, float value);

    /**
     * Get the <code>float</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    float get(const std::string &key, float default_value) const;

    /**
     * Assign a <code>double</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, double value);

    /**
     * Get the <code>double</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    double get(const std::string &key, double default_value) const;

    /**
     * Assign a <code>std::string</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager).
     */
    void set(const std::string &key, const std::string &value);

    /**
     * Get the <code>std::string</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::string get(const std::string &key,
                    const std::string &default_value) const;
    /**
     * Assign a <code>Variable</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, VariablePtr value);
    void set(const std::string &key, ExpressionPtr value);
    void set(const std::string &key, PayloadPtr value);
    void set(const std::string &key, IoPtr value);
    void set(const std::string &key, WaypointPtr value);

    /**
     * Get the <code>Variable</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    VariablePtr get(const std::string &key, VariablePtr default_value) const;
    ExpressionPtr get(const std::string &key,
                      ExpressionPtr default_value) const;
    PayloadPtr get(const std::string &key, PayloadPtr default_value) const;
    IoPtr get(const std::string &key, IoPtr default_value) const;
    WaypointPtr get(const std::string &key, WaypointPtr default_value) const;

    /**
     * Assign a <code>bool[]</code> as value to the specified key.
     * @param key key in the data model (not <code>null} and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager).
     */
    void set(const std::string &key, const std::vector<bool> &value);

    /**
     * Get the <code>bool[]</code> as value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::vector<bool> get(const std::string &key,
                          const std::vector<bool> &default_value) const;

    /**
     * Assign a <code>int[]</code> as value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, const std::vector<int> &value);

    /**
     * Get the <code>int[]</code> as value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::vector<int> get(const std::string &key,
                         const std::vector<int> &default_value) const;

    /**
     * Assign a <code>long[]</code> as value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null} or an empty
     * <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, const std::vector<long> &value);
    void set(const std::string &key, const std::vector<uint64_t> &value);

    /**
     * Get the <code>long[]</code> as value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::vector<long> get(const std::string &key,
                          const std::vector<long> &default_value) const;
    std::vector<uint64_t> get(const std::string &key,
                              const std::vector<uint64_t> &default_value) const;

    /**
     * Assign a <code>float[]</code> as value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, const std::vector<float> &value);

    /**
     * Get the <code>float[]</code> as value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::vector<float> get(const std::string &key,
                           const std::vector<float> &default_value) const;

    /**
     * Assign a <code>double[]</code> as value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, const std::vector<double> &value);

    /**
     * Get the <code>double[]</code> as value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::vector<double> get(const std::string &key,
                            const std::vector<double> &default_value) const;

    /**
     * Assign a <code>std::string[]</code> as value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, const std::vector<std::string> &value);

    /**
     * Get the <code>std::string[]</code> as value assigned to the specified
     * key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    std::vector<std::string> get(
        const std::string &key,
        const std::vector<std::string> &default_value) const;

    /**
     *  Assign a <code>TCP</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, TCPPtr value);

    /**
     * Get the <code>TCP value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    TCPPtr get(const std::string &key, TCPPtr default_value) const;

    /**
     *  Assign a <code>Feature</code> value to the specified key.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @param value value assigned to key.
     *
     * @throws IllegalArgumentException if the key is <code>null</code> or an
     * empty <code>std::string</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    void set(const std::string &key, FeaturePtr value);

    /**
     * Get the <code>Feature</code> value assigned to the specified key.
     *
     * @param key key in the data model.
     * @param default_value value to be returned, if key does not exist.
     * @return the value assigned to the key. If not exist, default_value is
     * returned.
     */
    FeaturePtr get(const std::string &key, FeaturePtr default_value) const;

    /**
     * Get a set of all the keys in the data model.
     *
     * @return A Set of keys.
     */
    std::set<std::string> getKeys() const;

    /**
     * Check if a key is present in the data model.
     *
     * @param key key in the data model (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @return <code>true</code>, if key exist, otherwise <code>false</code>.
     */
    bool isSet(const std::string &key) const;

    /**
     * Remove a key-value pair from the data model.
     *
     * @param key key in the data model  (not <code>null</code> and not an empty
     * <code>std::string</code>).
     * @return <code>true</code>, if succeed, otherwise <code>false</code>.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * program node outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    bool remove(const std::string &key);

private:
    friend class DataSwitch;
    DataModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif
