#include <aubo_caps/domain/extension_info.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(Device);

/**
 * This base interface represents a device supported by AuboScope, e.g. a
 * gripping device.
 */
class ARCS_ABI_EXPORT Device
{
public:
    virtual ~Device() = 0;
    /**
     * Get information about the AuboCap that added/contributed this gripper
     * device to AuboScope.
     *
     * @return AuboCap information for this gripper device
     */
    virtual AuboCapInfoPtr getContributorInfo() = 0;

    /**
     * <p>
     * Get the unique identifier (id) for this gripper. This can be used to
     * identify this specific gripper among the set of individual grippers
     * available in a specific multi-gripper device.
     * </p>
     *
     * Note that this id is not "globally" unique, but only unique for the
     * individual grippers belonging to a specific multi-gripper device.
     *
     * @return The unique identifier of this gripper
     */
    virtual std::string getId() = 0;

    /**
     * <p>
     * The name of the gripper displayed in the AuboScope UI, e.g. in the
     * Gripper toolbar and program node.
     * </p>
     *
     * <b>Note:</b> This name must not be used to uniquely identify a specific
     * individual gripper, since the name might be translated (depending on the
     * selected language in AuboScope) or be changed between versions of the
     * AuboCap that added the gripper.
     *
     * @return The name of this gripper displayed in AuboScope
     */
    virtual std::string getDisplayName() = 0;

    /**
     * A device cannot be guaranteed to be present in AuboScope. A device will
     * not be present if the AuboCap that added the device is not installed.
     * This method can be used to determine, if the device is present.
     *
     * @return <code>true</code> if this gripper device is present in AuboScope,
     * otherwise <code>false</code>.
     */
    virtual bool isResolvable() = 0;
};
} // namespace aubo_scope
} // namespace arcs
