#ifndef AUBO_SCOPE_FEATURE_H
#define AUBO_SCOPE_FEATURE_H

#include <vector>
#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(Feature);

/**
 * This base interface represents all types of features available in AuboScope
 */
class ARCS_ABI_EXPORT Feature
{
public:
    Feature(Feature &f);
    Feature(Feature &&f);
    virtual ~Feature();

    /**
     * Note: The feature can be renamed at any time
     *
     * @return The name of the feature
     */
    std::string getDisplayName();

    /**
     * Returns whether the position of the feature has been fully
     * defined/specified.
     *
     * @return <code>true</code> if the feature is defined, <code>false</code>
     * otherwise.
     */
    bool isDefined();

    /**
     * Returns whether the feature can be used as a variable in the program.
     *
     * @return <code>true</code> if the feature is variable, <code>false</code>
     * otherwise.
     */
    bool isVariable();
    bool isJoggable();

    /**
     * A feature cannot be guaranteed to be present in AuboScope. The end user
     * can load a different installation (not containing the feature) or the
     * feature can be removed by the end user or the AuboCap that added the
     * feature. This method can be used to determine, if the feature is present.
     *
     * @return <code>true</code> if this feature is present in AuboScope,
     * otherwise <code>false</code>.
     */
    bool isResolvable();

    /**
     * Returns the Cartesian location of the feature
     *
     * @return A pose representing the Cartesian location of the feature
     */
    std::vector<double> getPose();

    size_t hashCode();

protected:
    Feature();

private:
    friend class DataSwitch;
    void *d_{ nullptr };
};

struct FeatureDefinedFilter
{
    bool operator()(FeaturePtr feature)
    {
        return feature && feature->isDefined();
    }
};

struct FeatureJoggableFilter
{
    bool operator()(FeaturePtr feature)
    {
        return feature && feature->isJoggable() && feature->isDefined();
    }
};

} // namespace aubo_scope
} // namespace arcs
#endif
