#ifndef AUBO_SCOPE_FEATURE_CONTRIBUTION_MODEL_H
#define AUBO_SCOPE_FEATURE_CONTRIBUTION_MODEL_H

#include <aubo_caps/domain/feature/feature.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(FeatureContributionModel);

/**
 * <p>
 * This interface provides functionality for adding, updating and removing
 * features in AuboScope.
 * </p>
 *
 * <b>Note:</b> This interface must only be used in an installation
 * contribution/
 */
class ARCS_ABI_EXPORT FeatureContributionModel
{
public:
    FeatureContributionModel(FeatureContributionModel &f);
    FeatureContributionModel(FeatureContributionModel &&f);
    ~FeatureContributionModel();

    /**
     * Add a feature to the current AuboScope installation. This makes it
     * selectable by the end user. The feature is not modifiable by the end
     * user.
     *
     * @param idKey The key to identify this feature by. The key is for this
     * AuboCap only, i.e. it only has to be unique for this AuboCap and not
     * "globally" for other aubo_studio plugins.
     * @param suggestedName Suggested name for the feature. Valid names must
     * match regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters. The
     * final name can be retrieved from the returned feature instance.
     * @param pose The pose of the feature with respect to the robot base
     * @return The feature created and registered in AuboScope.
     * @throws FeatureAlreadyAddedException If a feature has previously been
     * added the same <code>idKey</code> identifier. Use {@link
     * #getFeature(String)} to check if the feature has already been added to
     * the current installation. Use {@link #updateFeature(String, Pose)} to
     * update the feature.
     * @throws IllegalFeatureNameException If the suggested name does not match
     * required regex.
     */
    FeaturePtr addFeature(const std::string &idKey,
                          const std::string &suggestedName,
                          const std::vector<double> &pose);
    /**
     * Returns the feature previously added by this AuboCap using the same
     * <code>idKey</code> identifier. Use this to verify if the feature is
     * present in the current installation.
     *
     * @param idKey The key to identify this feature by.
     * @return The feature previously added by this AuboCap.
     *         Returns <code>null</code> if no feature exists in the current
     * installation.
     */
    FeaturePtr getFeature(const std::string &idKey);

    /**
     * Update the pose of an existing feature added by this AuboCap.
     *
     * @param idKey The key to identify the feature. A feature must have been
     * added prior to this method call using the same key.
     * @param newPose The new pose to set for the feature.
     * @throws FeatureNotFoundException If no feature exists with the provided
     * <code>idKey</code>.
     */
    void updateFeature(const std::string &idKey,
                       const std::vector<double> &newPose);
    void renameFeature(const std::string &idKey, const std::string &newName);

    /**
     * Remove a feature added by this AuboCap from AuboScope. Program nodes
     * using the feature will be become undefined because the feature is no
     * longer resolvable.
     *
     * @param idKey The key used to add the feature with.
     * @throws FeatureNotFoundException If no feature exists with the provided
     * <code>idKey</code>.
     */
    void removeFeature(const std::string &idKey);

private:
    friend class DataSwitch;
    FeatureContributionModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_FEATURE_CONTRIBUTION_MODEL_H
