#ifndef AUBO_SCOPE_FUNCTION_MODEL_H
#define AUBO_SCOPE_FUNCTION_MODEL_H

#include <aubo_caps/domain/function/function.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(FunctionModel);

/**
 * <p>
 * This interface provides methods to add and remove script functions in the
 *Expression Editor accessible from Assignment and If program nodes among
 *others.
 * </p>
 *
 * <p>
 * Added functions merely serve as an easy way for the end customer to use the
 *contributed functions. The implementation of a given script function must be
 *added in the preamble of program through the use of
 * {@link InstallationNodeContribution#generateScript(ScriptWriter)}.
 * </p>
 *
 * <p>
 * The full set of functions which can be contributed must be added in the
 *preamble at all times, regardless of whether they are added to the Expression
 *Editor or not. Otherwise old programs that rely on a script function
 *contribution might stop working.
 *</p>
 *
 * See also {@link Function}.
 */
class ARCS_ABI_EXPORT FunctionModel
{
public:
    FunctionModel(FunctionModel &f);
    FunctionModel(FunctionModel &&f);
    ~FunctionModel();

    /**
     * @param name Name of the function. Safe names must match regex
     * [a-zA-Z][a-zA-Z0-9_]{0,24} for a total of 25 characters. A range of
     * Universal Robots reserved names exists and cannot be used. In case of an
     *             illegal or reserved name, an {@link FunctionException} will
     * be thrown.
     * @param argumentNames Is a list of hints (for the function arguments) to
     * be shown comma separated inside angle brackets in the Expression Editor.
     * No more than five arguments can be specified. The hints must also match
     * the regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters.
     *                      Otherwise an {@link FunctionException} will be
     * thrown.
     * @return If the script function was successfully added, an object
     * representing the Function will be returned. Null otherwise (e.g. if a
     * function with the same name was already added).
     * @throws FunctionException if the name is illegal or more than five
     * arguments are specified.
     */
    FunctionPtr addFunction(const std::string &name,
                            const std::vector<std::string> &argumentNames);

    /**
     * @param function Function to be removed. Only functions added by this
     * AuboCap can be removed.
     * @return <code>true</code>, if the function was successfully removed.
     *         <code>false</code> otherwise (e.g. if the function was not added
     * by this AuboCap).
     */
    bool removeFunction(FunctionPtr func);

    /**
     * @param name The name of the function.
     * @return The function if found, otherwise null.
     */
    FunctionPtr getFunction(const std::string &name);

    /**
     * @param filter The filter being applied to the full list of added
     * functions (by aubo_studio plugins).
     * @return A collection of added functions that meet the filter criteria.
     */
    std::vector<FunctionPtr> getFunctions();

private:
    friend class DataSwitch;
    FunctionModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_FUNCTION_MODEL_H
