#ifndef AUBO_SCOPE_IO_H
#define AUBO_SCOPE_IO_H

#include <string>
#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(Io);

/**
 *
 * This is the base interface representing all types of I/Os available in
 * AuboScope.
 */
class ARCS_ABI_EXPORT Io
{
public:
    enum IoType : int
    {
        DIGITAL,
        ANALOG,
        INTEGER,
        BOOLEAN,
        DOUBLE
    };
    enum InterfaceType : int
    {
        STANDARD,
        TOOL,
        CONFIGURABLE,
        LINKIO,
        MODBUS,
        GENERAL_PURPOSE,
    };

public:
    Io(Io &f);
    Io(Io &&f);
    virtual ~Io();

    /**
     *
     * @return the user defined name of the IO in the currently selected
     * installation.
     */
    std::string getDisplayName() const;

    /**
     *
     * @return the robot default name of the I/O.
     */
    std::string getDefaultName() const;

    /**
     *
     * @return get current reading of the I/O as a string.
     */
    std::string getValueStr() const;

    /**
     *
     * @return the I/O type.
     */
    IoType getType() const;

    /**
     *
     * @return the interface type this I/O belongs to.
     */
    InterfaceType getInterfaceType() const;

    /**
     *
     * @return the action number of the I/O.
     */
    int getAction() const;

    /**
     *
     * @return <code>true</code> if I/O supports reading of values and
     * <code>false</code> otherwise.
     */
    bool isInput() const;

    /**
     * Some I/Os, such as MODBUS I/Os and analog Tool inputs, are in some
     * situations not present in AuboScope or available for use. This method can
     * be used to determine if the I/O is available. <br>
     *
     * For more details about the specific situations where certain types of
     * I/Os can be unresolved, see the relevant interfaces extending this base
     * interface, e.g. {@link ModbusIO#isResolvable()}.
     *
     * @return <code>true</code>, if I/O can be resolved.
     *         <code>false</code> otherwise.
     */
    bool isResolvable() const;

    size_t hashCode();

protected:
    Io();

private:
    friend class DataSwitch;
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_IO_H
