#ifndef AUBO_SCOPE_MODBUS_IO_H
#define AUBO_SCOPE_MODBUS_IO_H

#include <aubo_caps/domain/io/io.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ModbusIo);

/**
 *
 * This interface provides support for MODBUS I/Os.
 */
class ARCS_ABI_EXPORT ModbusIo : public Io
{
public:
    ModbusIo(ModbusIo &f);
    ModbusIo(ModbusIo &&f);
    ~ModbusIo();

    /**
     *
     * @return Get the IP address for this MODBUS I/O.
     */
    std::string getIpAddress();

    /**
     *
     * @return the signal address for this MODBUS I/O.
     */
    std::string getSignalAddress();

    /**
     *
     * @param value Set MODBUS signal to an integer value. If the MODBUS signal
     * is digital, then <code>false</code> is represented by 0 and
     * <code>true</code> are represented by all numbers different from 0.
     *
     * @return <code>true</code> if the MODBUS signal was sent down to the
     * controller, and <code>false</code> if the signal was not sent down, e.g.
     * because the controller was not running. Notice that the MODBUS signal
     * must support writes, consult the Aubo MODBUS protocol.
     */
    bool setValue(int value);

    /**
     *
     * @return the integer value of the MODBUS signal. If the MODBUS signal is
     * digital then <code>false</code> is represented by 0 and
     * <code>true}</code> is represented by the value 1.
     */
    int getValue();

private:
    friend class DataSwitch;
    ModbusIo();
    void *d_{ nullptr };
};

struct ModbusIoFilter
{
    ModbusIoFilter(Io::IoType type, bool is_input)
        : type_(type), is_input_(is_input)
    {
    }
    bool operator()(IoPtr io)
    {
        return std::dynamic_pointer_cast<ModbusIo>(io) != nullptr &&
               io->getType() == type_ && io->isInput() == is_input_;
    }

private:
    Io::IoType type_;
    bool is_input_;
};

} // namespace aubo_scope
} // namespace arcs

#endif // MODBUSIO_H
