#ifndef AUBO_SCOPE_PAYLOAD_H
#define AUBO_SCOPE_PAYLOAD_H

#include <vector>
#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(Payload);

class ARCS_ABI_EXPORT Payload
{
public:
    Payload(Payload &f);
    Payload(Payload &&f);
    ~Payload();

    /**
     * <p>
     * This enum represents the different types of warnings that the parameters
     * of a payload can have.
     * </p>
     *
     * A payload can be usable, but may have a value for a payload parameter or
     * may have a combination of (some of the) payload parameters, that could
     * affect performance of the robot, if the payload is used (i.e., if the
     * parameters are applied).
     */
    enum ParametersWarning
    {

        /**
         * None of the parameters of the payload or the combination of payload
         * parameters has any warnings.
         */
        NO_WARNING,

        /**
         * The payload has a mass of 0 kg which is most likely not
         * intended/valid. Note that the value for the payload mass should be
         * the total mass of the payload attached to the tool output flange of
         * the robot.
         */
        MASS_0KG,

        /**
         * The mass of the payload exceeds the rated maximum supported payload
         * mass of the robot. Using the payload could affect performance of the
         * robot.
         */
        RATED_MASS_EXCEEDED,

        /**
         * The combination of the mass and the center of gravity of could affect
         * the performance of the robot, if the payload is used.
         */
        MASS_AND_CENTER_OF_GRAVITY_EXCEEDED,
    };

    /**
     * @return The total mass of the payload (attached to the tool output flange
     * of the robot)
     */
    double getMass() const;

    /**
     * Gets the payload's center of gravity (CoG), also referred to as center of
     * mass. It is defined as the offset between the center of the tool output
     * flange and the center of gravity of the attached payload.
     *
     * @return The center of gravity (CoG) of the payload relative to the center
     * of the tool output flange
     */
    std::vector<double> getCenterOfGravity() const;

    /**
     * <p>
     * Gets the inertia matrix of the payload. The inertia matrix is defined in
     * a coordinate system with center at the payload's center of gravity (CoG)
     * and the axes aligned with the tool output flange coordinate system
     * (axes).
     * </p>
     *
     * <b>Note:</b> The inertia matrix specified by the end user can be invalid.
     * The {@link #isUsable} method can be used determine, if the payload can be
     * used (i.e., has a valid inertia matrix).
     *
     * @return The inertia matrix of the payload
     */
    std::vector<double> getInertiaMatrix() const;

    /**
     * <p>
     * Gets the name of the payload displayed in the AuboScope UI.
     * </p>
     *
     * <b>Note:</b> The payload can change name at any time, and thus the name
     * must not be used to uniquely identify a specific payload. The payload
     * name can change in the following situations: <ul> <li> The payload is
     * renamed by the end user </li> <li> The name gets translated (depending on
     * the selected language in AuboScope) </li> <li> The name is changed
     * between versions of the AuboCap that added the payload</li>
     * </ul>
     *
     * @return The name of the payload displayed in AuboScope
     */
    std::string getDisplayName() const;

    /**
     * <p>
     * A payload cannot be guaranteed to be present in AuboScope. This method
     * can be used to determine, if the payload is present.
     * </p>
     *
     * <p>
     * The payload will not be present, if the end user loads a different
     * installation which does not contain the payload, or if the payload is
     * removed by the end user or the AuboCap that added the payload.
     * </p>
     *
     * <b>Note:</b> Storing an undefined payload in a program node
     * contribution's (a {@link ProgramNodeContribution} instance) data model
     * ({@link DataModel} interface) will automatically make the program node
     * (and the program) undefined.
     *
     * @return <code>true</code> if this payload is present in AuboScope,
     * otherwise <code>false</code>.
     */
    bool isResolvable();

    /**
     * <p>
     * Use this method to determine if the payload can be used. An unresolvable
     * payload (see the {@link #isResolvable} method) will always be unusable.
     * </p>
     *
     * <p>
     * A payload can have an invalid inertia matrix, where all the individual
     * values of the matrix are within the valid range, but the combination of
     * values is invalid.
     * </p>
     *
     * <b>Note:</b> Storing an unusable payload in a program node contribution's
     * (a {@link ProgramNodeContribution} instance) data model ({@link
     * DataModel} interface) will automatically make the program node (and the
     * program) undefined.
     *
     * @return <code>true</code>, if the payload is usable, otherwise
     * <code>false</code>. <code>false</code> will always be returned when the
     * payload is unresolvable (see the {@link #isResolvable} method).
     */
    bool isUsable();

    /**
     * <p>
     * Use this method to check if the parameters of this payload have any
     * warnings.
     * </p>
     *
     * <p>
     * A payload can be usable, but may have a value for a payload parameter or
     * may have a combination of (some of the) payload parameters, that could
     * affect performance of the robot, if the payload is used (i.e., if the
     * parameters are applied).
     * </p>
     *
     * An unresolvable payload (see {@link #isResolvable()}) will have no
     * warning for its parameters (i.e.
     * {@link ParametersWarning#NO_WARNING} is returned).
     *
     * @return The warning (if any) for the parameters of this payload.
     */
    ParametersWarning getParametersWarning();

    size_t hashCode();

private:
    friend class DataSwitch;
    Payload();
    void *d_{ nullptr };
};
} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_PAYLOAD_H
