#ifndef AUBO_SCOPE_PAYLOAD_CONTRIBUTION_MODEL_H
#define AUBO_SCOPE_PAYLOAD_CONTRIBUTION_MODEL_H

#include <aubo_caps/class_forward.h>
#include <aubo_caps/domain/payload/payload.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(PayloadContributionModel);

class ARCS_ABI_EXPORT PayloadContributionModel
{
public:
    PayloadContributionModel(PayloadContributionModel &f);
    PayloadContributionModel(PayloadContributionModel &&f);
    ~PayloadContributionModel();

    /**
     * <p>
     * Add a payload to the current installation in AuboScope.
     * </p>
     *
     * <p>
     * This makes the payload selectable by the end user as well as available to
     * other AuboCap contributions (through the
     * {@link PayloadModel} interface). The payload is not modifiable by the end
     * user or other aubo_studio plugins.
     * </p>
     *
     * <p>
     * A default guess for the inertia matrix for the payload will be computed
     * based on the specified payload mass and center of gravity. If data for
     * the payload's inertia matrix is available, the method
     * {@link #addPayload(String, String, Mass, Position, InertiaMatrix)} can
     * used instead (this might give better performance of the robot arm).
     * </p>
     *
     * <p>
     * <b>Note:</b> The specified payload mass must be the total mass of the
     * payload attached to the tool output flange of the robot.
     * </p>
     *
     * The valid range for the payload mass and the coordinates of the center of
     * gravity (CoG) are available through the
     * {@link RobotLimits} interface which can be accessed with {@link
     * RobotModel#getRobotLimits()} (an instance of the
     * {@link RobotModel} interface can be retrieved through {@link
     * SystemAPI#getRobotModel()}).
     *
     * @param idKey The key to identify this payload by, not <code>null</code>
     * nor an empty string. The key is for this AuboCap only, i.e. it only has
     * to be unique for this AuboCap and not "globally" for other aubo_studio
     * plugins.
     * @param suggestedName Suggested name for the payload, not
     * <code>null</code> nor an empty string. Valid names must match regex
     * [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters. The final name
     * can be retrieved from the returned payload instance.
     * @param mass The total mass of the payload attached to the tool output
     * flange of the robot, not <code>null</code>. Create it using {@link
     * SimpleValueFactory#createMass(double, Mass.Unit)}.
     * @param centerOfGravity The center of gravity (CoG), also referred to as
     * center of mass, for the payload. It is defined as the offset between the
     * center of the tool output flange and the center of gravity of the
     * attached payload. Cannot be <code>null</code>. Create it using the {@link
     * PositionFactory} interface.
     * @return The payload created and registered in AuboScope.
     * @throws PayloadAlreadyAddedException If a payload has previously been
     * added using the same <code>idKey</code> identifier. Use {@link
     * #getPayload(String)} to check if the payload has already been added to
     * the current installation. Use {@link #updatePayload(String, Mass,
     * Position)} or
     *                                      {@link #updatePayload(String, Mass,
     * Position, InertiaMatrix)} to update the payload if the payload parameters
     * have changed.
     * @throws IllegalPayloadNameException If the suggested name does not match
     * required regex.
     * @throws IllegalMassException If the specified mass is not inside the
     * valid range as defined by AuboScope.
     * @throws IllegalCenterOfGravityException If any of the values of the
     * specified center of gravity are not inside the valid range as defined by
     * AuboScope.
     */
    PayloadPtr addPayload(const std::string &id_key,
                          const std::string &suggested_name, double mass,
                          const std::vector<double> &center_of_gravity);

    /**
     * <p>
     * Add a payload to the current installation in AuboScope.
     * </p>
     *
     * <p>
     * This makes the payload selectable by the end user as well as available to
     * other AuboCap contributions (through the
     * {@link PayloadModel} interface). The payload is not modifiable by the end
     * user or other aubo_studio plugins.
     * </p>
     *
     * Use the {@link #addPayload(String, String, Mass, Position,
     * InertiaMatrix)} method instead if data for the payload's inertia matrix
     * is not available.
     *
     * <p>
     * <b>Note:</b> The specified payload mass must be the total mass of the
     * payload attached to the tool output flange of the robot.
     * </p>
     *
     * The valid range for the payload mass and the coordinates of the center of
     * gravity (CoG) are available through the
     * {@link RobotLimits} interface which can be accessed with {@link
     * RobotModel#getRobotLimits()} (an instance of the
     * {@link RobotModel} interface can be retrieved through {@link
     * SystemAPI#getRobotModel()}).
     *
     * @param idKey The key to identify this payload by, not <code>null</code>
     * nor an empty string. The key is for this AuboCap only, i.e. it only has
     * to be unique for this AuboCap and not "globally" for other aubo_studio
     * plugins.
     * @param suggestedName Suggested name for the payload, not
     * <code>null</code> nor an empty string. Valid names must match regex
     * [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters. The final name
     * can be retrieved from the returned payload instance.
     * @param mass The total mass of the payload attached to the tool output
     * flange of the robot, not <code>null</code>. Create it using {@link
     * SimpleValueFactory#createMass(double, Mass.Unit)}.
     * @param centerOfGravity The center of gravity (CoG), also referred to as
     * center of mass, for the payload. It is defined as the offset between the
     * center of the tool output flange and the center of gravity of the
     * attached payload. Cannot be <code>null</code>. Create it using the {@link
     * PositionFactory} interface.
     * @param inertiaMatrix The inertia matrix of the payload, defined in a
     * coordinate system with center at the payload's center of gravity and the
     * axes aligned with the tool output flange coordinate system (axes). Cannot
     * be <code>null</code>. Create it using
     *                      {@link #createInertiaMatrix(double, double, double,
     * double, double, double, MomentOfInertia.Unit)}.
     * @return The payload created and registered in AuboScope.
     * @throws PayloadAlreadyAddedException If a payload has previously been
     * added using the same <code>idKey</code> identifier. Use {@link
     * #getPayload(String)} to check if the payload has already been added to
     *                                  the current installation.
     *                                  Use {@link #updatePayload(String, Mass,
     * Position)} or
     *                                  {@link #updatePayload(String, Mass,
     * Position, InertiaMatrix)} to update the payload if the payload parameters
     * have changed.
     * @throws IllegalPayloadNameException If the suggested name does not match
     * required regex.
     * @throws IllegalMassException If the specified mass is not inside the
     * valid range as defined by AuboScope.
     * @throws IllegalCenterOfGravityException If any of the values of the
     * specified center of gravity are not inside the valid range as defined by
     * AuboScope.
     * @throws IllegalInertiaMatrixException If any of the values of the
     * specified inertia matrix are not inside the valid range as defined by
     * AuboScope.
     */
    PayloadPtr addPayload(const std::string &id_key,
                          const std::string &suggested_name, double mass,
                          const std::vector<double> &center_of_gravity,
                          const std::vector<double> &inertiaMatrix);

    /**
     * Returns the payload previously added by this AuboCap using the same
     * <code>idKey</code> identifier. Use this to verify if the payload is
     * present in the current installation.
     *
     * @param idKey The key to identify this payload by, not <code>null</code>
     * nor an empty string.
     * @return The payload previously added by this AuboCap.
     *         Returns <code>null</code> if no payload exists in current
     * installation (with the specified <code>idKey</code> identifier).
     */
    PayloadPtr getPayload(const std::string &id_key);

    /**
     * <p>
     * Update the mass and center of gravity (CoG) of an existing payload added
     *by this AuboCap.
     * </p>
     *
     * <b>Note:</b> If an inertia matrix has been specified for the payload,
     *calling this method will replace (overwrite) it with a computed default
     *guess based the new specified payload mass and center of gravity. To avoid
     *overwriting an existing inertia matrix (or to update an existing inertia
     *matrix), instead use the method
     *{@link #updatePayload(String, Mass, Position, InertiaMatrix)}.
     *
     * @param idKey The key to identify the payload, not <code>null</code> nor
     *an empty string. A payload must have been added using the same key prior
     *to calling this method.
     * @param mass The total mass of the payload attached to the tool output
     *flange of the robot, not <code>null</code>. Create it using {@link
     *SimpleValueFactory#createMass(double, Mass.Unit)}.
     * @param centerOfGravity The center of gravity of the payload, not {@code
     *null}. Create it using the
     *                        {@link PositionFactory} interface.
     * @throws PayloadNotFoundException If no payload exists with the provided
     *<code>idKey</code> identifier in the current installation.
     * @throws IllegalMassException If the specified mass is not inside the
     *valid range as defined by AuboScope.
     * @throws IllegalCenterOfGravityException If any of the values of the
     *specified center of gravity are not inside the valid range as defined by
     *AuboScope.
     */

    void updatePayload(const std::string &id_key, double mass,
                       const std::vector<double> &center_of_gravity);

    /**
     * Update the mass, center of gravity (CoG) and inertia matrix of an
     * existing payload added by this AuboCap.
     *
     * @param idKey The key to identify the payload, not <code>null</code> nor
     * an empty string. A payload must have been added using the same key prior
     * to calling this method.
     * @param mass The total mass of the payload attached to the tool output
     * flange of the robot, not <code>null</code>. Create it using {@link
     * SimpleValueFactory#createMass(double, Mass.Unit)}.
     * @param centerOfGravity The center of gravity of the payload, not {@code
     * null}. Create it using the
     *                        {@link PositionFactory} interface.
     * @param inertiaMatrix The inertia matrix of the payload, not
     * <code>null</code>. Create it using
     *                      {@link #createInertiaMatrix(double, double, double,
     * double, double, double, MomentOfInertia.Unit)}
     * @throws PayloadNotFoundException If no payload exists with the provided
     * <code>idKey</code> identifier in the current installation.
     * @throws IllegalMassException If the specified mass is not inside the
     * valid range as defined by AuboScope.
     * @throws IllegalCenterOfGravityException If any of the values of the
     * specified center of gravity are not inside the valid range as defined by
     * AuboScope.
     * @throws IllegalInertiaMatrixException If any of the values of the
     * specified inertia matrix are not inside the valid range as defined by
     * AuboScope.
     */
    void updatePayload(const std::string id_key, double mass,
                       const std::vector<double> &center_of_gravity,
                       const std::vector<double> &inertia_matrix);

    /**
     * Remove a payload added by this AuboCap from AuboScope. Program nodes
     * using the payload will be become undefined because the payload is no
     * longer resolvable.
     *
     * @param idKey The identifier key used to add the payload with, not {@code
     * null} nor an empty string.
     * @throws PayloadNotFoundException If no payload exists with the provided
     * <code>idKey</code> identifier in the current installation.
     */
    void removePayload(const std::string &id_key);

private:
    friend class DataSwitch;
    PayloadContributionModel();
    void *d_{ nullptr };
};
} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_PAYLOAD_CONTRIBUTION_MODEL_H
