#ifndef AUBO_SCOPE_GRIPPER_NODE_H
#define AUBO_SCOPE_GRIPPER_NODE_H

#include <aubo_caps/domain/device/gripper_device.h>
#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/undoredo/undo_redo_manager.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(GripperNode);

/**
 * <p>
 * This interface represents a program node which can be used for programming
 * grip and release actions with a selected gripper device.
 * </p>
 *
 * <p>
 * An instance of this interface can be built using the {@link
 * GripperNodeFactory} interface and is returned as a result when {@link
 * GripperNodeFactory#createGripperNode(GripperDevice)} is called.
 * </p>
 */
class ARCS_ABI_EXPORT GripperNode : public ProgramNode
{
public:
    /**
     * The configuration type used to determine which type of configuration this
     * instance is.
     */
    enum ConfigType
    {

        /**
         * <p>
         * A Gripper action (i.e. grip or release) has not been selected
         * </p>
         *
         * The config instance can be cast to {@link UndefinedActionConfig}.
         * This is only relevant if the Gripper node uses a multi-gripper
         * device, i.e. a gripper device supporting the multi-gripper capability
         * (see the
         * {@link MultiGripperSupport} interface). The instance will provide
         * access to the individual gripper selection.
         */
        UNDEFINED,

        /**
         * <p>
         * The gripper is configured for a grip action
         * </p>
         *
         * The config instance can be cast to {@link GripActionConfig}.
         */
        GRIP_ACTION,

        /**
         * <p>
         * The gripper is configured for a release action
         * </p>
         *
         * The config instance can be cast to {@link ReleaseActionConfig}.
         */
        RELEASE_ACTION
    };

    GripperNode(GripperNode &f);
    GripperNode(GripperNode &&f);
    virtual ~GripperNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return The type of this config.
     */
    ConfigType getConfigType();

    void setConfigType(ConfigType type);

    /**
     * @return The gripper device used by this Gripper node
     */
    GripperDevicePtr getGripperDevice();

    /**
     * The setting type used to determine which type of setting this instance
     * is.
     */
    enum SettingType
    {

        /**
         * <p>
         * No selection has been made.
         * </p>
         */
        UNDEFINED0,

        /**
         * <p>
         * A payload (from the installation) is selected.
         * </p>
         *
         * The setting instance can be cast to {@link PayloadSelection}.
         */
        PAYLOAD_SELECTION,

        /**
         * <p>
         * Custom payload parameters are specified.
         * </p>
         *
         * The setting instance can be cast to {@link CustomPayloadParameters}.
         */
        CUSTOM_PARAMETERS
    };

    /**
     * This method returns the type of setting. Cast this instance appropriately
     * to have access to specific getters.
     *
     * @return The type of this payload setting.
     */
    SettingType getSettingType();

private:
    friend class DataSwitch;
    GripperNode();
    void *d_{ nullptr };
};
} // namespace aubo_scope
} // namespace arcs

#endif
