#ifndef AUBO_SCOPE_LOOP_NODE_H
#define AUBO_SCOPE_LOOP_NODE_H

#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/value/expression/expression.h>
#include <aubo_caps/domain/variable/variable.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(LoopNode);

class ARCS_ABI_EXPORT LoopNode : public ProgramNode
{
public:
    /**
     * The configuration type used to determine which type of configuration this
     * instance is.
     */
    enum ConfigType
    {

        /**
         * <p>
         * Looping infinitely has been selected.
         * </p>
         *
         * The config instance can be cast to {@link AlwaysLoopNodeConfig}
         */
        ALWAYS,

        /**
         * <p>
         * Looping a certain number of times has been selected.
         * </p>
         *
         * The config instance can be cast to {@link CounterLoopNodeConfig}
         */
        COUNTER,

        /**
         * <p>
         * Looping as long as the evaluation of an expression is true has been
         * selected.
         * </p>
         *
         * The config instance can be cast to {@link ExpressionLoopNodeConfig}
         */
        EXPRESSION
    };

    LoopNode(LoopNode &f);
    LoopNode(LoopNode &&f);
    virtual ~LoopNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return the type of this config.
     */
    ConfigType getConfigType();
    void setConfigType(ConfigType type);

    /**
     * Creates a loop counter configuration.
     *
     * @param loopCount the number of iterations through the loop node.
     * @param errorHandler error handler for handling validation. If using
     * {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid loop count value.
     * @return the configuration.
     */
    void setLoopCount(int loop_count);
    int getLoopCount();

    /**
     * Create a loop expression configuration.
     *
     * @param expression the expression whose evaluation must be true to
     * continue looping, not <code>null</code>.
     * @param evaluateContinuously if <code>true</code> the expression will be
     * evaluated continuously during the looping (i.e. the loop can be
     * interrupted anytime during its execution), otherwise <code>false</code>
     * and the expression will be evaluated once per loop iteration.
     * @return the configuration.
     */
    void setExpression(ExpressionPtr expression);
    ExpressionPtr getExpression();

    /**
     *
     * @return <code>true</code> if the expression will be evaluated
     * continuously during the looping (i.e. the loop can be interrupted anytime
     * during its execution), otherwise <code>false</code> and the expression
     * will be evaluated once per loop iteration.
     */
    bool isEvaluateContinously();
    void setEvaluateContinously(bool cont);

    /**
     *
     * @return the variable for storing the loop count
     */
    void setLoopCountVariable(VariablePtr var);
    VariablePtr getLoopCountVariable();

private:
    friend class DataSwitch;
    LoopNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_LOOP_NODE_H
