#ifndef AUBO_SCOPE_MOVE_NODE_H
#define AUBO_SCOPE_MOVE_NODE_H

#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/feature/feature.h>
#include <aubo_caps/domain/tcp/tcp.h>
#include <aubo_caps/domain/variable/feature_variable.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(MoveNode);

class ARCS_ABI_EXPORT MoveNode : public ProgramNode
{
public:
    enum ConfigType : int
    {

        /**
         * <p>
         * Movement type is MoveJ.
         * </p>
         *
         * The config instance can be cast to {@link MoveJMoveNodeConfig}.
         */
        MOVE_J = 0,

        /**
         * <p>
         * Movement type is MoveL.
         * </p>
         *
         * The config instance can be cast to {@link MoveLMoveNodeConfig}.
         */
        MOVE_L,

        MOVE_S,
    };

    enum TcpSelectionType : int
    {

        /**
         * Motions are adjusted to the currently active TCP. The TCP is
         * determined during runtime of the program.
         */
        USE_ACTIVE_TCP = 0,

        /**
         * No TCP is used and the motion under this Move command will be with
         * respect to the tool output flange.
         */
        IGNORE_ACTIVE_TCP,

        /**
         * A specific TCP is selected. <br>
         * The TCPSelection instance can be cast to {@link MoveNodeTCP}.
         */
        MOVE_TCP
    };

    enum FeatureSelectionType : int
    {
        /**
         * The selected feature is no longer available (e.g. because the user
         * has removed the feature in the Installation or the needed
         * Installation is not currently loaded)
         */
        UNRESOLVED_FEATURE,

        /**
         * Feature is selected.
         * The Selection instance can be cast to {@link MoveNodeFeature}
         */
        FEATURE
    };

    MoveNode(MoveNode &f);
    MoveNode(MoveNode &&f);
    virtual ~MoveNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return the type of this config.
     */
    ConfigType getConfigType();
    void setConfigType(ConfigType type);

    bool getUseJoints();
    void setUseJoints(bool use);

    /**
     * <p>
     * Set joint speed parameter for the MoveJ configuration.
     * </p>
     *
     * The parameter applies to the movement of the robot arm, from the previous
     * position through the waypoints under the Move node.
     *
     * @param joint_speed The shared joint speed to be achieved, not
     * <code>null</code>.
     * @param jointSpeedErrorHandler Error handler for handling validation. If
     * using {@link ErrorHandler#AUTO_CORRECT}, this will clamp the value to the
     * nearest valid joint speed value.
     * @return This builder.
     */
    void setJointSpeed(double joint_speed);
    double getJointSpeed();

    /**
     * <p>
     * Set joint acceleration parameter for the MoveJ configuration.
     * </p>
     *
     * The parameter applies to the movement of the robot arm, from the previous
     * position through the waypoints under the Move node.
     *
     * @param joint_acceleration The shared joint acceleration to be used, not
     * <code>null</code>.
     * @param joint_accelerationErrorHandler Error handler for handling
     * validation. If using {@link ErrorHandler#AUTO_CORRECT}, this will clamp
     * the value to the nearest valid joint acceleration value.
     * @return This builder.
     */
    void setJointAcceleration(double joint_acceleration);
    double getJointAcceleration();

    /**
     * <p>
     * Set the TCP selection for the MoveJ configuration.
     * </p>
     *
     * <p>
     * The TCP will be set as the active TCP before the movement of the robot
     * arm, from the previous position through the waypoints under the Move
     * node, is performed.
     * </p>
     *
     * See {@link TCPSelection} which can be created using the {@link
     * TCPSelectionFactory} interface.
     *
     * @param tcp_selection The TCP setting to be applied.
     * @return This builder.
     */
    void setTCPSelection(TCPPtr tcp_selection);
    TCPPtr getTCPSelection();

    /**
     * <p>
     * Set tool speed parameter for the MoveL configuration.
     * </p>
     *
     * The parameter applies to the movement of the robot arm, from the previous
     * position through the waypoints under the Move node.
     *
     * @param tool_speed The shared tool speed to be achieved, not
     * <code>null</code>.
     * @param toolSpeedErrorHandler Error handler for handling validation. If
     * using {@link ErrorHandler#AUTO_CORRECT}, this will clamp the value to the
     * nearest valid tool speed value.
     * @return This builder.
     */
    void setToolSpeed(double tool_speed);
    double getToolSpeed();

    /**
     * <p>
     * Set tool acceleration parameter for the MoveL configuration.
     * </p>
     *
     * The parameter applies to the movement of the robot arm, from the previous
     * position through the waypoints under the Move node.
     *
     * @param tool_acceleration The shared tool acceleration to be used, not
     * <code>null</code>.
     * @param toolAccelerationErrorHandler Error handler for handling
     * validation. If using {@link ErrorHandler#AUTO_CORRECT}, this will clamp
     * the value to the nearest valid tool acceleration value.
     * @return This builder.
     */
    void setToolAcceleration(double tool_acceleration);
    double getToolAcceleration();

    /**
     * <p>
     * Set feature for the MoveL configuration.
     * </p>
     *
     * The feature will be used for the movement of the robot arm, from the
     * previous position through the waypoints under the Move node.
     *
     * @param feature The shared feature for this movement, not
     * <code>null</code>.
     * @return This builder.
     */
    void setFeature(FeaturePtr feature);
    void setFeature(FeatureVariablePtr feature_variable);
    FeaturePtr getFeature();

    /**
     * <p>
     * Set the blend parameter for the MoveP configuration.
     * </p>
     *
     * The blend parameter will be used between all waypoints in the movement of
     * the robot arm, from the previous position through the waypoint under the
     * Move node.
     *
     * @param blend The shared blend used in the motion of this Move node, not
     * <code>null</code>.
     * @param blendErrorHandler Error handler for handling validation. If using
     * {@link ErrorHandler#AUTO_CORRECT}, this will clamp the value to the
     * nearest valid blend value.
     * @return This builder.
     */
    void setBlend(double blend);
    double getBlend();

    void setTcpSelectionType(TcpSelectionType type);
    TcpSelectionType getTcpSelectionType();

private:
    friend class DataSwitch;
    MoveNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_MOVE_NODE_H
