#ifndef AUBO_SCOPE_SET_NODE_H
#define AUBO_SCOPE_SET_NODE_H

#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/value/expression/expression.h>
#include <aubo_caps/domain/io/io.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(SetNode);

class ARCS_ABI_EXPORT SetNode : public ProgramNode
{
public:
    /**
     * The configuration type used to determine which type of configuration this
     * instance is.
     */
    enum ConfigType
    {
        /**
         * No selection has been made. This type has no further information.
         */
        NO_TYPE,

        /**
         * No action has been selected. This type has no further information.
         */
        NO_ACTION,

        /**
         * <p>
         * Digital output has been selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * DigitalOutputSetNodeConfig}.
         */
        DIGITAL_OUTPUT,

        /**
         * <p>
         * Analog output configured for electric current or no output has been
         * selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * AnalogOutputCurrentSetNodeConfig}.
         */
        ANALOG_OUTPUT_CURRENT,

        /**
         * <p>
         * Analog output configured for voltage has been selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * AnalogOutputVoltageSetNodeConfig}.
         */
        ANALOG_OUTPUT_VOLTAGE,

        /**
         * <p>
         * Register using float type values has been selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * FloatRegisterOutputSetNodeConfig}.
         */
        FLOAT_REGISTER_OUTPUT,

        /**
         * <p>
         * Expression output has been selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * ExpressionOutputSetNodeConfig}.
         */
        EXPRESSION_OUTPUT,

        /**
         * <p>
         * Single digital pulse output has been selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * SingleDigitalPulseSetNodeConfig}.
         */
        SINGLE_DIGITAL_PULSE,

        /**
         * <p>
         * Increment variable has been selected. This type has no further
         * information.
         * </p>
         *
         * Setting this type of config will be ignored and will have no effect
         * on the node.
         */
        INCREMENT_VARIABLE,
    };

    enum OutputSelectionType
    {

        /**
         * No output is selected.
         */
        NO_OUTPUT,

        /**
         * The selected output is no longer available (e.g. because the end user
         * has removed the output in the Installation or the needed Installation
         * is not currently loaded).
         */
        UNRESOLVED_OUTPUT,

        /**
         * <p>
         * Output is selected.
         * </p>
         *
         * The Selection instance can be cast to {@link ExpressionOutput}.
         */
        OUTPUT
    };
    SetNode(SetNode &f);
    SetNode(SetNode &&f);
    virtual ~SetNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return the type of this config.
     */
    ConfigType getConfigType();
    void setConfigType(ConfigType type);

    /**
     * @return the time for the length of the pulse.
     */
    double getPulseTime();
    void setPulseTime(double time);

    /**
     *
     * @return the float value to be applied to the float register output.
     */
    float getFloatValueToSet();
    void setFloatValueToSet(float value);

    /**
     * @return the analog output value.
     */
    double getAnalogOutput();
    void setAnalogOutput(double value);

    OutputSelectionType getOutputSelectionType();

    /**
     *
     * @return the expression whose evaluation will be applied to the output.
     */
    ExpressionPtr getExpression();
    void setExpression(ExpressionPtr expression);

    /**
     * The output selected can be an analog output, a digital output, a MODBUS
     * output or a register output.
     *
     * @return the output to which the result of the evaluation of the
     * expression will be applied.
     */
    IoPtr getOutput();
    void setOutput(IoPtr io);

    /**
     * Depending on the type of output this is interpreted in different ways.
     * For digital output and MODBUS output,
     * <code>true</code> means the output will be set to HIGH,
     * </code>false</code> means the output will be set to LOW. For boolean
     * register the value is interpreted literally.
     *
     * @return the output value.
     */
    bool getBoolValueToSet();
    void setBoolValueToSet(bool value);

    /**
     *
     * @return the voltage value/level to be applied to the output
     */
    double getVoltage();
    void setVoltage(double voltage);

    /**
     *
     * @return the electric current value/level to be applied to the output.
     */
    double getCurrent();
    void setCurrent(double curr);

private:
    friend class DataSwitch;
    SetNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_SET_NODE_H
