#ifndef AUBO_SCOPE_UNTIL_NODE_H
#define AUBO_SCOPE_UNTIL_NODE_H

#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/value/expression/expression.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(UntilNode);

/**
 * <p>
 * The Until node is a type of program node that must be under a Waypoint
 * node or Direction node.
 * </p>
 *
 * <p>
 * It has special behavior when calling {@link
 * TreeNode#addChild(ProgramNode)} and {@link
 * TreeNode#removeChild(TreeNode)}. Until nodes by default are leaf nodes,
 * however they can have actions added to them. When that happens they are
 * transformed into branch nodes that allow child nodes. When the last child
 * is removed they become leaf nodes again. This is all automatically
 * handled when {@link TreeNode#addChild(ProgramNode)} and {@link
 * TreeNode#removeChild(TreeNode)} are called.
 * </p>
 */
class ARCS_ABI_EXPORT UntilNode : public ProgramNode
{
public:
    /**
     * The configuration type used to determine which type of configuration this
     * instance is.
     */
    enum ConfigType : short
    {

        /**
         * <p>
         * An Until node where the until condition is true once the
         * evaluation of an expression is true.
         * </p>
         *
         * The config instance can be cast to {@link
         * ExpressionUntilNodeConfig}.
         */
        EXPRESSION,
        /**
         * <p>
         * An Until node where the until condition is true once an impact on
         * the Tool is detected.
         * </p>
         *
         * This functionality is not supported by the API.
         */
        TOOL_CONTACT,
        /**
         * <p>
         * An Until node where the until condition is true once a given I/O
         * reaches a certain value.
         * </p>
         *
         * The config can be cast to {@link IOInputUntilNodeConfig}.
         */
        IO_INPUT,

        /**
         * <p>
         * An Until node where the until condition is true once a given
         * distance is reached.
         * </p>
         *
         * The config can be cast to {@link DistanceUntilNodeConfig}.
         */
        DISTANCE,
        /**
         * <p>
         * An Until node where the until condition is true once its parent
         * Waypoint is reached.
         * </p>
         *
         * The config can be cast to {@link ReachedWaypointUntilNodeConfig}.
         */
        REACHED_WAYPOINT,

        /**
         * An undefined Until node with no type set yet.
         */
        NONE_CONFIG_TYPE

    };

    /**
     * Compare operator types LESS_THAN and GREATER_THAN used when waiting for
     * an analog input to go lower or higher, respectively, than a threshold
     */
    enum CompareOperator : int
    {
        LESS_THAN,
        GREATER_THAN
    };

    /**
     * Blend Parameters can be of different types
     * The parameter type is used to determine the explicit blend parameters
     * (can be used in place of instanceof operations).
     */
    enum BlendParameterType
    {

        /**
         * The robot arm will stop without performing a blend.
         */
        NO_BLEND,

        /**
         * <p>
         * Custom blend has been selected.
         * </p>
         *
         * The blend instance can be cast to {@link CustomBlendParameters}.
         */
        CUSTOM_BLEND
    };

    enum CustomDecelerationParameterType
    {

        /**
         * This deceleration was either created specifying values for both
         * {@link AngularAcceleration} and {@link Acceleration} and/or is in use
         * in an Until node that is not yet in the program tree.
         */
        Unknown,

        /**
         * This deceleration is currently used under a MoveJ node.
         */
        Angular,

        /**
         * This deceleration is currently used under a MoveL or MoveP node.
         */
        Cartesian
    };

    /**
     * The deceleration parameter type used to determine which type of
     * deceleration parameter this instance is.
     */
    enum DecelerationParameterType
    {
        /**
         * Shared deceleration has been selected. The acceleration values from
         * the parent Move node will be used as deceleration (absolute value).
         */
        SHARED_DECELERATION,

        /**
         * <p>
         * Custom deceleration has been selected.
         * </p>
         *
         * This instance can be cast to {@link CustomDecelerationParameters}.
         */
        CUSTOM_DECELERATION
    };

    /**
     * The type of the input selection.
     */
    enum IOInputSelectionType
    {

        /**
         * The node is not fully configured.
         */
        NONE_INPUT_SELECT,

        /**
         * <p>
         * Digital input is selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * DigitalInputUntilNodeConfig}.
         */
        DIGITAL_INPUT,

        /**
         * <p>
         * Analog input is selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * AnalogInputUntilNodeConfig}.
         */
        ANALOG_INPUT,

        /**
         * <p>
         * A float input register is selected.
         * </p>
         *
         * The config instance can be cast to {@link
         * FloatRegisterInputUntilNodeConfig}.
         */
        FLOAT_REGISTER_INPUT
    };

    UntilNode(UntilNode &f);
    UntilNode(UntilNode &&f);
    virtual ~UntilNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return the type of this config.
     */
    ConfigType getConfigType();
    void setConfigType(ConfigType type);

    BlendParameterType getBlendParameterType();
    void setBlendParameterType(BlendParameterType type);

    /**
     * Creates blend parameters for an Until node. Can be used in a Distance
     * configuration type.
     *
     * @param suggestedBlend This blend value may be (automatically) corrected
     * depending on the value of the distance parameter the Distance
     * configuration.
     * @return the blend parameters for the Until node.
     */
    void setWaypointBlendParameter(double suggested_blend);
    double getWaypointBlendParameter();

    DecelerationParameterType getDecelerationParameterType();
    void setDecelerationParameterType(DecelerationParameterType type);

    /**
     * <p>
     * Creates deceleration parameter for an Until node. Can be used in an
     * Expression configuration type.
     * </p>
     *
     * An angular deceleration is used when the type of the parent Move node is
     * a MoveJ, whereas a Cartesian deceleration is used when the move type is a
     * MoveL or MoveP.
     *
     * @param deceleration the angular deceleration (the value for the Cartesian
     * deceleration will be the default value that AuboScope uses), not
     * <code>null</code>.
     * @param decelerationErrorHandler error handler for handling validation. If
     * using {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid angular deceleration value.
     * @return the deceleration parameter for the Until node.
     */
    void setDecelerationParameters(double deceleration);
    double getDecelerationParameters();

    /**
     *
     * @return the deceleration type. This can change depending on the Move node
     * type above the Until node.
     */
    CustomDecelerationParameterType getCustomDecelerationParameterType();

    void setCustomDecelerationParameterType(
        CustomDecelerationParameterType type);

    /**
     * The rate at which to decelerate when the until condition is met.
     *
     * @return the specified angular deceleration of the until node (absolute
     * value).
     */
    double getAngularDeceleration();
    void setAngularDeceleration(double dec);

    /**
     * The rate at which to decelerate when the until condition is met.
     *
     * @return the specified Cartesian deceleration of the until node (absolute
     * value).
     */
    double getCartesianDeceleration();
    void setCartesianDeceleration(double dec);

    /**
     * <p>
     * Creates an Expression configuration for an Until node with an expression
     * and default value for the deceleration parameter. The until condition for
     * this configuration is true once the evaluation of the expression is true.
     * </p>
     *
     * Note that this default value may change between AuboScope versions.
     *
     * @param expression the expression whose evaluation will determine whether
     * the until condition is met.
     * @return the Until node configuration.
     */
    void setExpressionConfig(ExpressionPtr expression);
    ExpressionPtr getExpressionConfig();

    /**
     * The distance the robot will travel.
     *
     * @return The distance.
     */
    double getDistance();
    void setDistance(double dis);
    IOInputSelectionType getIOInputSelectionType();
    void setIOInputSelectionType(IOInputSelectionType type);

    /**
     * Creates an I/O Input configuration for an Until node. The until condition
     * for this configuration is true once the signal value of a given digital
     * input goes HIGH or LOW.
     *
     * @param input the digital input whose signal value will determine whether
     * the until condition is met, not <code>null</code>.
     * @param valueToProceedUntil <code>true</code>, if the until condition is
     * met when the input signal value goes HIGH, <code>false</code> otherwise
     * (condition met when signal value is LOW).
     * @return the Until node configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values). See {@link IO#isInput()}.
     */
    void setDigitalInputValueToProceed(bool value_to_proceed_until);
    bool getDigitalInputValueToProceed();

    /**
     * Creates an I/O Input configuration for an Until node. The until condition
     * for this configuration is true once the the signal value of a given
     * analog input goes past a specified electric Current threshold.
     *
     * @param input the analog input whose signal value will determine whether
     * the until condition is met, not <code>null</code>.
     * @param compareOperator the operator to use when comparing the signal
     * value of the selected analog input to the threshold value, not
     * <code>null</code>. Available options are "less than" ({@literal <}) or
     * "greater than"
     * ({@literal >}).
     * @param currentThreshold the electric Current threshold for the signal
     * value of the analog input. Execution will proceed until the signal value
     * of the selected analog input goes past this threshold.
     * @param errorHandler the error handler for validation errors. If using
     * {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid current value.
     * @return the Until node configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values). See {@link IO#isInput()}.
     */

    void setAnalogInputValueToProceed(double value);
    double getAnalogInputValueToProceed();

    void setCompareOperator(CompareOperator compare_operator);
    CompareOperator getCompareOperator();

private:
    friend class DataSwitch;
    UntilNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_UNTIL_NODE_H
