#ifndef AUBO_SCOPE_WAIT_NODE_H
#define AUBO_SCOPE_WAIT_NODE_H

#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/io/analog_io.h>
#include <aubo_caps/domain/io/digital_io.h>
#include <aubo_caps/domain/io/modbus_io.h>
#include <aubo_caps/domain/io/boolean_register.h>
#include <aubo_caps/domain/io/double_register.h>
#include <aubo_caps/domain/value/expression/expression.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(WaitNode);

class ARCS_ABI_EXPORT WaitNode : public ProgramNode
{
public:
    /**
     * The configuration type used to determine which type of configuration this
     * instance is.
     */
    enum ConfigType : int
    {

        NO_TYPE,

        /**
         * <p>
         * Nothing to wait.
         * </p>
         *
         *
         */
        NO_WAIT,

        /**
         * <p>
         * Wait type is time.
         * </p>
         *
         * The config instance can be cast to {@link TimeWaitNodeConfig}.
         */
        TIME,

        /**
         * <p>
         * Wait type is digital_input.
         * </p>
         *
         * The config instance can be cast to {@link
         * DigitalInputWaitNodeConfig}.
         */
        DIGITAL_INPUT,

        /**
         * <p>
         * Wait type is analog_input_current.
         * </p>
         *
         * The config instance can be cast to {@link
         * AnalogInputCurrentWaitNodeConfig}.
         */
        ANALOG_INPUT_CURRENT,

        /**
         * <p>
         * Wait type is analog_input_voltage.
         * </p>
         *
         * The config instance can be cast to {@link
         * AnalogInputVoltageWaitNodeConfig}.
         */
        ANALOG_INPUT_VOLTAGE,

        /**
         * <p>
         * Wait type is float_register_input.
         * </p>
         *
         * The config instance can be cast to {@link
         * FloatRegisterInputWaitNodeConfig}.
         */
        FLOAT_REGISTER_INPUT,

        /**
         * <p>
         * Wait type is expression_input.
         * </p>
         *
         * The config instance can be cast to {@link
         * ExpressionInputWaitNodeConfig}.
         */
        EXPRESSION_INPUT
    };

    /**
     * Compare operator types LESS_THAN and GREATER_THAN used when waiting for
     * an analog input to go lower or higher, respectively, than a threshold
     */
    enum CompareOperator : int
    {
        LESS_THAN,
        GREATER_THAN
    };

    WaitNode(WaitNode &f);
    WaitNode(WaitNode &&f);
    virtual ~WaitNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return the type of this config.
     */
    ConfigType getConfigType();
    void setConfigType(ConfigType type);

    /**
     *
     * @return the time to wait for. second
     */
    double getWaitTime();
    void setWaitTime(double time);

    /**
     * Depending on the type of input this is interpreted in different ways.
     * For digital inputs and MODBUS inputs, <code>true</code> means the input
     * to wait for must be HIGH, <code>false</code> means the input to wait for
     * must be LOW. For boolean registers the value is interpreted literally.
     *
     * @return the input value to wait for.
     */
    bool getBoolValueToWaitFor();
    void setBoolValueToWaitFor(bool value);

    float getFloatValueToWaitFor();
    void setFloatValueToWaitFor(float value);

    /**
     * Creates a configuration for waiting for a digital input to go high or
     * low.
     *
     * @param input the digital input on which to wait, not <code>null</code>.
     * @param value_to_wait_for <code>true</code>, if waiting for the input
     * signal value/level to go HIGH, <code>false</code> otherwise (waiting for
     * LOW signal value/level).
     * @return the configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values). See {@link IO#isInput()}.
     */
    void setDigitalInput(DigitalIoPtr input);
    DigitalIoPtr getDigitalInput();

    /**
     * Creates a configuration for waiting for a digital MODBUS input to go high
     * or low.
     *
     * @param input the digital MODBUS input on which to wait, not
     * <code>null</code>.
     * @param value_to_wait_for <code>true</code>, if waiting for the input
     * signal value/level to go HIGH, <code>false</code> otherwise (waiting for
     * LOW signal value/level).
     * @return the configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values) or if <code>output</code> is not a digital
     * output. See {@link IO#isInput()} and {@link IO#getType()}.
     */
    void setModbusInput(ModbusIoPtr input);
    ModbusIoPtr getModbusInput();

    /**
     * Creates a configuration for waiting for a boolean register to go
     * <code>true</code> or <code>false</code>.
     *
     * @param input the boolean register input on which to wait, not
     * <code>null</code>.
     * @param value_to_wait_for <code>true</code>, if waiting for the register
     * to be <code>true}, <code>false} otherwise.
     * @return the configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values). See {@link IO#isInput()}.
     */
    void setBoolRegister(BoolRegisterPtr input);
    BoolRegisterPtr getBoolRegister();

    void setCompareOperator(CompareOperator compareOperator);
    CompareOperator getCompareOperator();

    /**
     * Creates a configuration for waiting for an analog input to go past an
     * electric current threshold.
     *
     * @param input on which to wait, not <code>null</code>.
     * @param compareOperator the operator to use when comparing input to value,
     * not <code>null</code>. Available options are "less than" ({@literal <})
     * or "greater than" ({@literal >}).
     * @param current the electric current threshold to wait for, not
     * <code>null</code>.
     * @param errorHandler the error handler for validation errors. If using
     * {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid current value.
     * @return the configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values). See {@link IO#isInput()}.
     */
    void setAnalogInput(AnalogIoPtr input);
    AnalogIoPtr getAnalogInput();

    /**
     * Creates a configuration for waiting for a register to go past a float
     * threshold.
     *
     * @param input the float register input on which to wait, not
     * <code>null</code>.
     * @param compareOperator the operator to use when comparing register to
     * value, not <code>null</code>. Available options are "less than"
     * ({@literal <}) or "greater than" ({@literal >}).
     * @param value_to_wait_for the float threshold to wait for.
     * @return the configuration.
     * @throws IllegalArgumentException if <code>input</code> is an output (does
     * not support reading of values). See {@link IO#isInput()}.
     */
    void setDoubleRegisterInput(DoubleRegisterPtr input);
    DoubleRegisterPtr getDoubleRegisterInput();

    /**
     * Create a configuration for waiting for an expression to be evaluated to
     * true.
     *
     * @param expression the expression to evaluate, not <code>null</code>.
     * @return the configuration.
     */
    void setExpressionInputConfig(ExpressionPtr expression);
    ExpressionPtr getExpressionInputConfig();

private:
    friend class DataSwitch;
    WaitNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_WAIT_NODE_H
