#ifndef AUBO_SCOPE_WAYPOINT_NODE_H
#define AUBO_SCOPE_WAYPOINT_NODE_H

#include <vector>
#include <aubo_caps/domain/program/nodes/program_node.h>
#include <aubo_caps/domain/variable/variable.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(WaypointNode);

class ARCS_ABI_EXPORT WaypointNode : public ProgramNode
{
public:
    /**
     * The configuration type used to determine which type of configuration this
     * instance is.
     */
    enum ConfigType : int
    {
        /**
         * <p>
         * Fixed position has been selected (fixed waypoint).
         * </p>
         *
         * The config instance can be cast to {@link
         * FixedPositionWaypointNodeConfig}.
         */
        FIXED_POSITION = 0,

        /**
         * <p>
         * Variable position has been selected (variable waypoint).
         * </p>
         *
         * The config instance can be cast to {@link
         * VariablePositionWaypointNodeConfig}.
         */
        VARIABLE_POSITION,

        /**
         * Relative position has been selected (relative waypoint).
         */
        RELATIVE_POSITION
    };

    /**
     * The blend parameters type used to determine which type of blend
     * parameters this instance is.
     */
    enum BlendType : int
    {
        /**
         * No blend has been selected (the robot arm will stop at this point).
         */
        NO_BLEND = 0,

        /**
         * Shared blend has been selected.
         */
        SHARED_BLEND,

        /**
         * <p>>
         * Waypoint blend has been selected.
         * </p>
         *
         * The blend instance can be cast to {@link WaypointBlendParameters}.
         */
        WAYPOINT_BLEND
    };

    /**
     * The blend parameters type used to determine which type of blend
     * parameters this instance is.
     */
    enum MotionType : int
    {
        /**
         * <p>
         * Time motion parameters has been selected.
         * </p>
         *
         * This instance can be cast to {@link TimeMotionParameters}.
         */
        TIME_MOTION,

        /**
         * <p>
         * Joint motion parameters has been selected.
         * </p>
         *
         * This instance can be cast to {@link JointMotionParameters}.
         */
        JOINT_MOTION,

        /**
         * <p>
         * Tool motion parameters has been selected.
         * </p>
         *
         * This instance can be cast to {@link ToolMotionParameters}.
         */
        TOOL_MOTION,

        /**
         * Shared motion parameters has been selected.
         */
        SHARED_MOTION
    };

    enum VariableType : int
    {
        /**
         * This variable type pose parameters has been selected.
         */
        VARIABLE_POSE,

        /**
         * This variable type joint parameters has been selected.
         */
        VARIABLE_JOINT
    };

    enum RelativePositionType : int
    {
        /**
         * The teaching location has been selected.
         */
        TEACHING,

        /**
         * The input offset has been selected.
         */
        INPUT
    };

    enum RelativeOffsetType : int
    {
        /**
         * Custom offset has been selected.
         */
        CUSTOM,

        /**
         * Variable type has been selected.
         */
        VARIABLE
    };

    enum class PositionDefinition : int
    {
        /**
         * The position has not been defined.
         */
        UNDEFINED,

        /**
         * <p>
         * The position of the waypoint has been defined.
         * </p>
         *
         * Cast this instance to {@link FixedPositionDefinedWaypointNodeConfig}.
         */
        DEFINED
    };

    enum VariableDefinition : int
    {
        /**
         * A variable has not been selected. This instance has no further
         * information.
         */
        UNDEFINED,

        /**
         * <p>
         * A variable has been selected making this config defined.
         * </p>
         *
         * Cast this instance to {@link
         * VariablePositionDefinedWaypointNodeConfig}.
         */
        DEFINED
    };

    WaypointNode(WaypointNode &f);
    WaypointNode(WaypointNode &&f);
    virtual ~WaypointNode();

    /**
     * This method returns the type of configuration. Cast this instance
     * appropriately to have access to specific getters.
     *
     * @return the type of this config.
     */
    ConfigType getConfigType();

    /**
     * Get blend type of the waypoint
     * @return the blend type.
     */
    BlendType getBlendType();

    MotionType getMotionType();

    /**
     * Cast this instance appropriately to have access to specific getters.
     *
     * @return the position definition.
     */
    PositionDefinition getPositionDefinition();

    /**
     * Cast this instance appropriately to have access to specific getters.
     *
     * @return the variable definition.
     */
    VariableDefinition getVariableDefinition();

    /**
     * <p>
     * Get the blend to be used for the movement.
     * </p>
     *
     * Note that, if the configuration has not been applied to a waypoint node
     * in the program tree, the blend value returned is a "suggested" value,
     * since the blend may be (automatically) corrected depending on surrounding
     * waypoints in the program tree (when it is applied to a waypoint node).
     *
     * @return the blend for the waypoint.
     */
    double getBlend();

    /**
     * @return the joint speed to be achieved.
     */
    double getJointSpeed();

    /**
     * @return the joint acceleration to use for the movement.
     */
    double getJointAcceleration();

    /**
     * @return the time for the waypoint movement.
     */
    double getTime();

    /**
     * @return the tool speed to be achieved.
     */
    double getToolSpeed();

    /**
     * @return the tool acceleration to use for the movement
     */
    double getToolAcceleration();

    /**
     * @return the waypoint position and orientation.
     */
    std::vector<double> getPose();

    /**
     * @return the joint positions corresponding to the pose of the robot (when
     * taking the TCP offset into account)
     */
    std::vector<double> getJointPositions();

    /**
     * @return the pose for the TCP offset (used when defining the robot
     * position)
     */
    std::vector<double> getTcpOffset();

    /**
     * @return the variable for the waypoint.
     */
    VariablePtr getVariable();

    /**
     * <p>
     * Creates shared motion parameters for a waypoint.
     * </p>
     *
     * Can be used for waypoints under a MoveJ, MoveL or a MoveP.
     *
     * @return the motion parameters for the waypoint.
     */
    void setSharedMotionParameters();

    /**
     * <p></p>
     * Creates time motion parameters for a waypoint.
     * </p>
     *
     * Can be used for waypoints under a MoveJ or a MoveL.
     *
     * @param duration the time the movement should take, not <code>null</code>.
     * @param errorHandler error handler for handling validation. If using
     * {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid time value.
     * @return the time motion parameters for the waypoint.
     */
    void setTimeMotionParameters(unsigned int duration);

    /**
     * <p>
     * Creates joint motion parameters for a waypoint.
     * </p>
     *
     * Can be used for waypoints under a MoveJ.
     *
     * @param jointSpeed the joint speed to be achieved, not <code>null</code>.
     * @param jointSpeedErrorHandler error handler for handling validation. If
     * using {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid joint speed value.
     * @param joint_acceleration the joint acceleration to be used for the
     * movement, not <code>null</code>.
     * @param joint_accelerationErrorHandler error handler for handling
     * validation. If using {@link ErrorHandler#AUTO_CORRECT} this will clamp
     * the value to the nearest valid joint acceleration value.
     * @return the joint motion parameters for the waypoint.
     */
    void setJointMotionParameters(double joint_speed,
                                  double joint_acceleration);

    /**
     * <p>
     * Creates tool motion parameters for a waypoint.
     * </p>
     *
     * Can be used for waypoints under a MoveL or a MoveP.
     *
     * @param tool_speed the tool speed to be achieved, not <code>null</code>.
     * @param tool_speedErrorHandler error handler for handling validation. If
     * using {@link ErrorHandler#AUTO_CORRECT} this will clamp the value to the
     * nearest valid tool speed value.
     * @param tool_acceleration the tool acceleration to be used for the
     * movement, not <code>null</code>.
     * @param tool_accelerationErrorHandler error handler for handling
     * validation. If using {@link ErrorHandler#AUTO_CORRECT} this will clamp
     * the value to the nearest valid tool acceleration value.
     * @return the tool motion parameters for the waypoint.
     */
    void setToolMotionParameters(double tool_speed, double tool_acceleration);

    /**
     * <p>
     * Creates a fixed position configuration with default values for a
     * waypoint.
     * </p>
     *
     * Note that these default values may change between AuboScope versions.
     *
     * @return the waypoint configuration.
     */
    void setFixedPositionConfig();

    /**
     * <p>
     * Creates a fixed position configuration for a waypoint with a defined
     * pose.
     * </p>
     *
     * @param pose_including_tcp the position and orientation for the waypoint,
     * not <code>null</code>.
     * @param qNear the joint positions that will be used for the inverse
     * kinematics calculations as a robot configuration near the waypoint pose,
     * not <code>null</code>.
     * @param blendParameters the blend parameters for the waypoint, not
     * <code>null</code>.
     * @param waypoint_motion_parameters the motion parameters for the waypoint,
     * not <code>null</code>.
     * @return the waypoint configuration.
     */
    void setFixedPositionConfig(const std::vector<double> &pose_including_tcp,
                                const std::vector<double> &qNear);

    /**
     * <p>
     * Creates a variable position configuration with default values for a
     * waypoint
     * </p>
     *
     * Note that these default values may change between AuboScope versions.
     *
     * @return the waypoint configuration.
     */
    void setVariablePositionConfig();

    /**
     * Creates a variable position configuration for a waypoint with a defined
     * variable
     *
     * @param variable the variable position and orientation for the waypoint,
     * not <code>null</code>.
     * @param blendParameters the blend parameters for the waypoint, not
     * <code>null</code>.
     * @param waypoint_motion_parameters the motion parameters for the waypoint,
     * not <code>null</code>.
     * @return the waypoint configuration.
     */
    void setVariablePositionConfig(VariablePtr variable);

private:
    friend class DataSwitch;
    WaypointNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_WAYPOINT_NODE_H
