#ifndef AUBO_SCOPE_PROGRAM_NODE_FACTORY_H
#define AUBO_SCOPE_PROGRAM_NODE_FACTORY_H

#include <aubo_caps/domain/program/nodes/builtin/assignment_node.h>
#include <aubo_caps/domain/program/nodes/builtin/circle_move_node.h>
#include <aubo_caps/domain/program/nodes/builtin/comment_node.h>
#include <aubo_caps/domain/program/nodes/builtin/direction_node.h>
#include <aubo_caps/domain/program/nodes/builtin/folder_node.h>
#include <aubo_caps/domain/program/nodes/builtin/force_node.h>
#include <aubo_caps/domain/program/nodes/builtin/halt_node.h>
#include <aubo_caps/domain/program/nodes/builtin/program_home_node.h>
#include <aubo_caps/domain/program/nodes/builtin/if_node.h>
#include <aubo_caps/domain/program/nodes/builtin/else_if_node.h>
#include <aubo_caps/domain/program/nodes/builtin/else_node.h>
#include <aubo_caps/domain/program/nodes/builtin/loop_node.h>
#include <aubo_caps/domain/program/nodes/builtin/move_node.h>
#include <aubo_caps/domain/program/nodes/builtin/pallet_node.h>
#include <aubo_caps/domain/program/nodes/builtin/popup_node.h>
#include <aubo_caps/domain/program/nodes/builtin/timer_node.h>
#include <aubo_caps/domain/program/nodes/builtin/screw_driving_node.h>
#include <aubo_caps/domain/program/nodes/builtin/script_node.h>
#include <aubo_caps/domain/program/nodes/builtin/seek_node.h>
#include <aubo_caps/domain/program/nodes/builtin/set_node.h>
#include <aubo_caps/domain/program/nodes/builtin/set_payload_node.h>
#include <aubo_caps/domain/program/nodes/builtin/until_node.h>
#include <aubo_caps/domain/program/nodes/builtin/wait_node.h>
#include <aubo_caps/domain/program/nodes/builtin/waypoint_node.h>
#include <aubo_caps/domain/program/nodes/builtin/switch_node.h>
#include <aubo_caps/domain/program/nodes/builtin/case_node.h>
#include <aubo_caps/domain/program/nodes/builtin/default_case_node.h>
#include <aubo_caps/domain/program/nodes/builtin/break_node.h>
#include <aubo_caps/domain/program/nodes/builtin/sub_prog_node.h>
#include <aubo_caps/domain/program/nodes/builtin/call_node.h>
#include <aubo_caps/domain/program/nodes/builtin/thread_node.h>
#include <aubo_caps/domain/program/nodes/builtin/offline_track_node.h>
#include <aubo_caps/domain/program/nodes/builtin/record_track_node.h>
#include <aubo_caps/domain/program/nodes/builtin/gripper_node.h>
#include <aubo_caps/domain/program/nodes/builtin/assembly_node.h>
#include <aubo_caps/domain/program/nodes/builtin/visual_localization_program_node.h>
#include <aubo_caps/domain/program/nodes/builtin/conveyor_tracking_node.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ProgramNodeFactory);
ARCS_CLASS_FORWARD(ProgramNodeService);

/**
 * <p>
 * This interfaces supplies methods to create different types of program nodes.
 * </p>
 *
 * <p>
 * Both built-in program nodes (provided by Universal Robots) and AuboCap
 * program nodes can be created. Most methods will create a node with the same
 * default configuration as if the end user added the node manually.
 * </p>
 *
 * <p>
 * Note that this default configuration may change between AuboScope versions.
 * </p>
 */
class ARCS_ABI_EXPORT ProgramNodeFactory
{
public:
    ProgramNodeFactory(ProgramNodeFactory &f);
    ProgramNodeFactory(ProgramNodeFactory &&f);
    virtual ~ProgramNodeFactory();

    /**
     * This method creates a AuboCap program node which is an instance of a
     * {@link contribution/ProgramNodeContribution}.
     *
     * @param typeid_service is the service creating the program node
     * contribution/
     *
     * @return the {@link ProgramNode} which represents the {@link
     * contribution/ProgramNodeContribution} created by {@link
     * ProgramNodeService}. Returns <code>null</code> if AuboCap providing
     * ProgramNodeService is not installed.
     *
     * @throws IllegalArgumentException if the argument ProgramNodeService
     * is <code>null</code>or ProgramNodeService does not implement {@link
     * ProgramNodeService}.
     */
    ProgramNodePtr createUserProgramNode(const char *typeid_service);

    /**
     * Creates a default Move node with one default Waypoint node under it.
     *
     *  @return new Move Node.
     */
    MoveNodePtr createMoveNode();

    /**
     * <p>
     * Creates a default Move node without a Waypoint node under it (e.g. to be
     * used for inserting named waypoints).
     * </p>
     *
     * Note that at least one Waypoint node must be under it for a Move node to
     * be valid.
     *
     * @return new Move Node.
     */
    MoveNodePtr createMoveNodeNoTemplate();

    /**
     * Creates a Waypoint node with default waypoint name (corresponds to the
     * name generated if the end user added a new node manually).
     *
     * @return new Waypoint Node.
     */
    WaypointNodePtr createWaypointNode();

    /**
     * <p>
     * Creates a Waypoint node with a suggested name.
     * </p>
     *
     * <p>
     * The final name can be different if the suggested name is already
     * registered. The waypoint name is registered when it is inserted into the
     * program tree.
     * </p>
     *
     * Variable waypoints will use the name of the variable instead of the
     * suggested name.
     *
     * @param suggested_name the suggested name of the waypoint. Valid names
     * must match regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters.
     * @return new Waypoint Node.
     * @throws IllegalWaypointNameException In case of an illegal name.
     */
    WaypointNodePtr createWaypointNode(const std::string &suggested_name);

    /**
     * Creates an Until Node with no until type specified.
     *
     * @return a new Until Node.
     */
    UntilNodePtr createUntilNode();
    DirectionNodePtr createDirectionNode();
    WaitNodePtr createWaitNode();
    SetNodePtr createSetNode();
    SetPayloadNodePtr createSetPayloadNode();
    PopupNodePtr createPopupNode();
    HaltNodePtr createHaltNode();
    CommentNodePtr createCommentNode();
    FolderNodePtr createFolderNode();
    LoopNodePtr createLoopNode();
    AssignmentNodePtr createAssignmentNode();
    IfNodePtr createIfNode();
    ElseIfNodePtr createElseIfNode();
    ElseNodePtr createElseNode();
    CircleMoveNodePtr createCircleMoveNode();
    PalletNodePtr createPalletNode();
    SeekNodePtr createSeekNode();
    ForceNodePtr createForceNode();
    AssemblyNodePtr createAssemblyNode();
    ScrewdrivingNodePtr createScrewdrivingNode();
    ScriptNodePtr createScriptNode();
    TimerNodePtr createTimerNode();
    SwitchNodePtr createSwitchNode();
    CaseNodePtr createCaseNode();
    DefaultCaseNodePtr createDefaultCaseNode();
    BreakNodePtr createBreakNode();
    SubProgNodePtr createSubProgNode();
    CallNodePtr createCallNode();
    OfflineTrackNodePtr createOfflineTrackNode();
    RecordTrackNodePtr createRecordTrackNode();
    ThreadNodePtr createThreadNode();
    ProgramHomeNodePtr createProgramHomeNode();
    VisualLocalizationProgramNodePtr createVisualLocalizationProgramNode();
    ConveyorTrackingNodePtr createConveyorTrackingNode();

    /**
     * <p>
     * This method creates a new Gripper program node. This node type can be
     * used for programming grip and release actions with a selected gripper
     * device.
     * </p>
     *
     * The program node will be created with the same default configuration as
     * if the end user added the node manually in AuboScope, i.e. an undefined
     * configuration where the gripper action is unselected.
     *
     * @param gripperDevice the gripper device to create a Gripper node for, not
     * <code>null</code>. The created Gripper node will program gripper actions
     * for this device.
     * @return a new Gripper program node (which can be inserted into the
     * program tree using the {@link TreeNode} interface).
     * @throws UnresolvedDeviceException when the specified
     * <code>gripperDevice</code> device is unresolved. See
     *                                   {@link GripperDevice#isResolvable()}.
     */
    GripperNodePtr createGripperNode(GripperDevicePtr gripperDevice);

private:
    friend class DataSwitch;
    ProgramNodeFactory();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif
