#ifndef AUBO_SCOPE_PROGRAM_MODEL_H
#define AUBO_SCOPE_PROGRAM_MODEL_H

#include <aubo_caps/domain/program/structure/tree_node.h>
#include <aubo_caps/contribution/program/program_node_contribution.h>
#include <aubo_caps/domain/program/nodes/program_node_factory.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ProgramModel);

/**
 * <h1>Summary</h1>
 * This interface makes it possible to modify sub-trees of aubo_studio plugins.
 * The structure of trees is separated from the contents of the nodes.
 * The {@link TreeNode} interface is concerned with the structure of trees
 * whereas the {@link ProgramNode} interface deals with the concrete content of
 * the nodes of the tree.
 *
 *
 *
 * <h1>Getting started with program modification</h1>
 * A ProgramModel contains a:
 * <ul>
 *     <li>getProgramNodeFactory() - that makes it possible to create new
 * program nodes (both built-in AuboScope nodes and AuboCap program nodes).
 * </li> <li>getRootTreeNode(ProgramNodeContribution root)} - returns the
 * sub-tree of a given AuboCap ProgramNodeContribution</li>
 * </ul>
 *
 *
 * The TreeNode interface supports insertion and removal of nodes. With it you
 * can retrieve the list of children of particular nodes. Calling
 * lockChildSequence on a tree node, locks the immediate children under the
 * node. I.e. children can not be rearranged, deleted or have other nodes
 * inserted into the child sequence by the end user.
 *
 * <h1>Example code</h1>
 * <pre>
 * public class UpdateProgramExampleContribution implements
 * ProgramNodeContribution { ProgramModel programModel;
 *
 *      public UpdateProgramExampleContribution(API api) {
 *          this.programModel = api.getProgramModel();
 *      }
 *
 *      private void insertIntoProgramTree() {
 *          TreeNode treeNode = programModel.createRootTreeNode(this);
 *          try {
 *              treeNode.addChild(programModel.getProgramNodeFactory().createCommentNode());
 *          } catch (TreeStructureException e) {
 *              // Your handler code
 *          }
 *      }
 *
 *      //...
 *      //... rest of the implementation and a call to insertIntoProgramTree()
 *      //...
 * }
 * </pre>
 */
class ARCS_ABI_EXPORT ProgramModel
{
public:
    ProgramModel(ProgramModel &f);
    ProgramModel(ProgramModel &&f);
    virtual ~ProgramModel();

    /**
     * This method returns a {@link ProgramNodeFactory} to create program nodes.
     *
     * @return the factory to create program nodes.
     */
    ProgramNodeFactoryPtr getProgramNodeFactory();

    /**
     * Gets the {@link TreeNode} root from {@link ProgramNodeContribution}. From
     * here, children can be added to form a sub-tree.
     *
     * @param root The AuboCap program node where a sub-tree is to be rooted.
     * @return Returns a {@link TreeNode} root.
     */
    TreeNodePtr getTreeNode(ProgramNodeContribution *root);

private:
    friend class DataSwitch;
    ProgramModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_PROGRAM_MODEL_H
