#ifndef AUBO_SCOPE_TREE_NODE_H
#define AUBO_SCOPE_TREE_NODE_H

#include <vector>
#include <functional>
#include <aubo_caps/domain/program/nodes/program_node.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(TreeNode);

/**
 * <p>
 * This interface represents a node in the program tree that can be used to
 * construct a sub-tree rooted in a AuboCap program node.
 * </p>
 *
 * Using the {@link ProgramModel#getRootTreeNode(ProgramNodeContribution)} to
 * obtain a root for the sub-tree, it is possible to add children. For each call
 * to {@link TreeNode#addChild(ProgramNode)}, a new {@link TreeNode} is
 * returned, that can, in turn, act as a root for yet another sub-tree.
 */
class ARCS_ABI_EXPORT TreeNode
{
public:
    TreeNode(TreeNode &f);
    TreeNode(TreeNode &&f);
    virtual ~TreeNode();

    /**
     * Add a child program node to the sub-tree.
     *
     * @param programNode the {@link ProgramNode} constructed using the {@link
     * ProgramNodeFactory}
     * @return Returns a TreeNode that can be used to add children to the newly
     * added child.
     * @throws TreeStructureException If it is not allowed to insert the {@link
     * ProgramNode} at this position a
     * {@link TreeStructureException} will be thrown.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    TreeNodePtr addChild(ProgramNodePtr program_node);

    /**
     * Inserts a child program node in the sub-tree directly before the existing
     * selected child node. Shifts the selected child node and any subsequent
     * nodes to positions after the newly added child.
     *
     * @param existingChildNode existing TreeNode child of this TreeNode.
     * @param programNode the {@link ProgramNode} constructed using the {@link
     * ProgramNodeFactory}.
     * @return Returns a TreeNode that can be used to add children to the newly
     * added child.
     * @throws TreeStructureException If it is not allowed to insert the {@link
     * ProgramNode} at this position or if the selected child node is not a
     * child of this TreeNode a {@link TreeStructureException} will be thrown.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    TreeNodePtr insertChildBefore(TreeNodePtr existingChildNode,
                                  ProgramNodePtr program_node);

    /**
     * Inserts a child program node under in the sub-tree directly after the
     * existing selected child node. Shifts any subsequent nodes to positions
     * after the newly added child.
     *
     * @param existingChildNode existing TreeNode child of this TreeNode.
     * @param programNode the {@link ProgramNode} constructed using the {@link
     * ProgramNodeFactory}.
     * @return Returns a TreeNode that can be used to add children to the newly
     * added child.
     * @throws TreeStructureException If it is not allowed to insert the {@link
     * ProgramNode} at this position or if the selected child node is not a
     * child of this TreeNode a {@link TreeStructureException} will be thrown.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    TreeNodePtr insertChildAfter(TreeNodePtr existingChildNode,
                                 ProgramNodePtr program_node);

    /**
     * Cut a child program node under in the sub-tree directly after the
     * existing selected child node. Shifts any subsequent nodes to positions
     * after the newly added child.
     *
     * @param child existing TreeNode child of this TreeNode.

     * @return Returns the TreeNode that has been cut off.
     * @throws TreeStructureException If it is not allowed to cut the {@link
     * ProgramNode} or if the selected child node is not a child of this
     * TreeNode
     * a {@link TreeStructureException} will be thrown.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    TreeNodePtr cutChildNode(TreeNodePtr child);

    /**
     * Removes a child node from the sub-tree. Be aware that removing the last
     * child will trigger the insertion of an
     * {@literal <empty>} child node.
     *
     * @param child The TreeNode child to be removed.
     * @return Returns <code>true</code> if removed successfully.
     * <code>false</code> otherwise.
     * @throws TreeStructureException If the removed child would leave the tree
     * in an illegal state a
     * {@link TreeStructureException} will be thrown.
     * @throws IllegalStateException if called from a Swing-based AuboCap
     * outside of an {@link UndoableChanges} scope (see also {@link
     * UndoRedoManager}).
     */
    bool removeChild(TreeNodePtr program_node);

    /**
     * @return a list of <code>TreeNode</code> objects that represents all the
     * children of this <code>TreeNode</code> (both programmatically added as
     * well as inserted by the end user).
     */
    std::vector<TreeNodePtr> getChildren();

    std::vector<TreeNodePtr> getChildrenNotSuppressed();

    /**
     * @return Parent treenode of the node
     */
    TreeNodePtr getParent();

    /**
     * @return Returns the {@link ProgramNode} at this position in the sub-tree.
     * Can either be a built-in AuboScope program node (provided by AUBO
     * Robots) or a AuboCap program node.
     */
    ProgramNodePtr getProgramNode();
    bool isSuppressed();

    /**
     * Whether the node is in the program tree; nodes not in the program tree
     * may have been cut or deleted.
     *
     * @return Returns <code>true</code> The node is in the program tree.
     * <code>false</code> The node is not in the program tree; such nodes may
     * have been cut or deleted.
     */
    bool isInProgramTree();

    /**
     * <p>
     * Gets a corresponding {@link TreeNode} instance for a child program node
     * (a {@link ProgramNode} instance) in the sub-tree under this {@link
     * TreeNode}.
     * </p>
     *
     * This method can for instance be used to gain access to the sub-tree under
     * a child program node encountered when iterating the sub-tree of the
     * parent node using the {@link ProgramNodeVisitor} (or any derived sub
     * class thereof).
     *
     * @param programNode program node to get a corresponding {@link TreeNode}
     * representation for, not <code>null</code>. Can either be a built-in
     * AuboScope program node (provided by Universal Robots) or a AuboCap
     * program node.
     * @return the <code>TreeNode</code> instance for the specified program
     * node.
     * @throws ProgramNodeNotInSubTreeException when the program node cannot be
     * found because it is not in the sub-tree.
     */
    TreeNodePtr locateDescendantTreeNode(ProgramNodePtr program_node);

    /**
     * Configures whether or not child nodes can be rearranged, deleted or have
     * other nodes inserted into the child sequence by the end user.
     *
     * @param isChildSequenceLocked If <code>true</code> then the immediate
     * children under this <code>TreeNode</code> will be locked.
     */
    void setChildSequenceLocked(bool isChildSequenceLocked);

    /**
     * <p>
     * This method traverses the entire sub-tree under this tree node in a
     * depth-first fashion (this corresponds to a top-down approach in the
     * program tree).
     * </p>
     *
     * <p>
     * A node visitor is used for callbacks to the visit-overloads you choose to
     * override. The overload called depends on the node type encountered.
     * Override the overloads for the node types you are concerned with. All
     * visit-methods have the program node, sibling index and depth as arguments
     * to help filter nodes if needed.
     * </p>
     *
     * <p>
     * The node visitor can be either a {@link ProgramNodeVisitor}
     * implementation with optional overrides or a
     * {@link ProgramNodeInterfaceVisitor} implementation. In the latter
     * case, the
     * {@link ProgramNodeInterfaceVisitor#visitAs(Object, int, int)}
     * method must be implemented.
     * </p>
     *
     * <p>
     * The {@link ProgramNodeInterfaceVisitor} can be used when targeting
     * AuboCap program nodes implementing the (generic) type parameter specified
     * in {@link ProgramNodeInterfaceVisitor} (see also
     * {@link ProgramNode#getAs(Class)}).
     * </p>
     *
     * Note that this method is sometimes called <code>accept()</code> in the
     * Visitor software design pattern.
     *
     * @param nodeVisitor the instance callbacks are made to.
     *
     * @return 0 for normal exit.
     *         non-zero for exception or error causing premature exit.
     */
    int traverse(std::function<int(ProgramNodePtr, int, int)> nodeVisitor);

private:
    friend class DataSwitch;
    TreeNode();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_TREE_NODE_H
