#ifndef AUBO_SCOPE_SCRIPT_WRITER_H
#define AUBO_SCOPE_SCRIPT_WRITER_H

#include <vector>
#include <string>
#include <aubo_caps/domain/variable/variable.h>
#include <aubo_caps/domain/value/expression/expression.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(ScriptWriter);

/**
 * This interface provides support for generating Script code.
 */
class ARCS_ABI_EXPORT ScriptWriter
{
public:
    ScriptWriter(ScriptWriter &f);
    ScriptWriter(ScriptWriter &&f);
    virtual ~ScriptWriter();

    void setLabel(int lineno, const std::string &comment);

    /**
     * Adds a single line of script code using auto-indentation.
     *
     * @param script_line single line of script code to append.
     */
    void appendLine(const std::string &script_line);
    void appendVectorDouble(const std::vector<double> &value);

    /**
     * Adds script code appending the script code as is without using
     * auto-indentation.
     *
     * @param script script code to append.
     */
    void appendRaw(const std::string &script);

    /**
     * Generate a string with the full script code.
     *
     * @return the resulting script as a string.
     */
    std::string generateScript();

    /**
     * <p>
     * Use this method when generating script code for a node that has children
     * nodes.globalVariable
     * </p>
     *
     * A simple example:
     * <pre>
     * {@code
     *    writer.ifCondition(expression)
     *    writer.writeChildren()          // let children nodes generate code.
     *    writer.end()
     * @endcode}
     * </pre>
     *
     * In most cases you prglobalVariableobably only want to call {@link
     * ScriptWriter#writeChildren()} once, but it is possible to call it
     * multiple times and thus have children nodes generate their code multiple
     * times.
     */
    void writeChildren();

    void assign(VariablePtr variable, const std::string &expression,
                bool sync = false);

    /**
     * <p>Variable assignment.
     * Assigns the variable without a local or global qualifier.
     * See the Script Manual for the scoping rules.</p>
     * @param variableName name of the variable, not null.
     * @param expression expression that is assigned to the variable, not null.
     */
    void assign(const std::string &variableName,
                const ExpressionPtr &expression, bool sync = false);

    /**
     * <p>Variable assignment.</p>
     * @param variable the variable to assign an expression to, not null.
     * @param expression expression that is assigned to the variable, not null.
     */
    void assign(VariablePtr variable, const ExpressionPtr &expression,
                bool sync = false);

    /**
     * Add 1 to the variable value.
     *
     * @param variable the variable to increment. Not <code>null</code>.
     */
    void incrementVariable(const std::string &variable_name);

    /**
     * Add a note.
     *
     * @param expression the note expression.
     */
    void note(const std::string &expression);

    /**
     * Sleep for a number of seconds.
     *
     * @param seconds amount of time to sleep in seconds.
     */
    void sleep(double seconds);

    /**
     * Uses up the remaining "physical" time a thread has in the current frame.
     */
    void sync();

    /**
     *
     * @param name Define a function of name name.
     */
    void defineFunction(const std::string &func_name);
    void anonyFunction(const std::string &func_name);

    void setRobotIndex(int robot_index);

    /**
     * Return from method.
     */
    void returnMethod();

    /**
     * Insert an end.
     */
    void end();

    /**
     * Insert an empty line.
     */
    void lineFeed();

    /**
     * <p>
     * Sets the mass and Center of Gravity (CoG) of the payload.
     * </p>
     *
     * This function must be called, when the payload weight or weigh
     * distribution changes significantly, i.e when the robot arm picks up or
     * puts down a heavy workpiece.
     *
     * @param payloadMass in kilograms
     * @param centerOfGravityX displacement from the tool-mount in meters.
     * @param centerOfGravityY displacement from the tool-mount in meters.
     * @param centerOfGravityZ displacement from the tool-mount in meters.
     */
    void setPayload(double mass, double x, double y, double z);

    /**
     * <p>
     * Set the Tool Center Point (TCP).
     * </p>
     *
     * Sets the transformation from the tool output flange coordinate system to
     * the TCP as a pose.
     *
     * @param x Position part.
     * @param y Position part.
     * @param z Position part.
     * @param rx Rotation part.
     * @param ry Rotation part.
     * @param rz Rotation part.
     */
    void setTcp(const std::vector<double> &pose);

    /**
     * Start an if-conditional.
     *
     * @param expression the expression of the if-sentence. Not
     * <code>null</code>.
     */
    void ifCondition(const ExpressionPtr &expression);

    /**
     * Start a negated if-conditional
     *
     * @param expression the expression of the negated if-sentence.
     */
    void ifNotCondition(const ExpressionPtr &expression);

    /**
     * Adds an else-if branch.
     *
     * @param expression the expression of the "else-if"-sentence.
     */
    void elseIfCondition(const ExpressionPtr &expression);

    /**
     * Adds an else branch.
     *
     */
    void elseCondition();

    /**
     * Starts a for-loop with tow loop invariants.
     *
     * @param count the loop counts.
     * @param step the loop step.
     */
    void forCondition(int count, int step);

    /**
     * Starts a while true loop.
     *
     */
    void whileTrue();

    /**
     * Starts a while-loop with a loop invariant.
     *
     * @param expression the loop invariant.
     */
    void whileCondition(const ExpressionPtr &expression);

    /**
     * Starts a while-loop with a negated loop invariant.
     *
     * @param expression the loop invariant that will be negated.
     */
    void whileNot(const ExpressionPtr &expression);

    /**
     * Start a thread definition with a given thread name.
     *
     * @param threadName the name of the new thread.
     * @param loop_or_not the new thread loops or not.
     */
    void defineThread(const std::string &thread_name, bool loop_or_not);

    /**
     * Start a previously defined thread.
     *
     * @param threadName The name of the thread that will be started.
     */
    void runThread(const std::string &thread_name);

    /**
     * @brief killThread
     * @param thread_name
     */
    void killThread(const std::string &thread_name);

    /**
     * <p>
     * Returns a registered variable name that can be used in a script.
     * </p>
     *
     * <p>
     * A variable is registered if it has been stored in a {@link DataModel}
     * instance or used for the configuration of a built-in AuboScope program
     * node.
     * </p>
     *
     * <b>Please note:</b> The name of a variable in the script can be different
     * from the value of {@link Variable#getDisplayName()}. You should not use
     * the value of {@link Variable#getDisplayName()} in a script.
     *
     * @param variable a registered Variable. Not <code>null</code>.
     * @return variable name that can be used in a script.
     * @exception IllegalArgumentException if the <code>variable</code> was not
     * registered in the data model or used for the configuration of a built-in
     * AuboScope program node.
     * @exception ClassCastException if the <code>variable</code> was not
     * created by
     * {@link domain/variable/VariableFactory}.
     *
     */
    std::string getResolvedVariable(const std::string &variable_name);

    void increaseIndent();
    void decreaseIndent();

private:
    friend class DataSwitch;
    ScriptWriter();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_SCRIPT_WRITER_H
