#ifndef AUBO_SCOPE_SYSTEM_API_H
#define AUBO_SCOPE_SYSTEM_API_H

#include <string>
#include <vector>
#include <unordered_map>
#include <aubo_caps/class_forward.h>

namespace arcs {
namespace aubo_scope {

enum class UnitType
{
    METRIC,  ///< 公制 mm Kg
    IMPERIAL ///< 英制
};

struct IoConfigure
{
    std::string label;
    int num;
    std::vector<std::string> names;
    std::vector<int> sequences;
    std::vector<int> functions;
    std::vector<int> safety_functions;
    std::vector<int> safety_functions_fixed;
    bool function_lock{ false };
    std::vector<int> types;
    std::vector<double> mins;
    std::vector<double> maxs;
};

ARCS_CLASS_FORWARD(SystemApi);
class ARCS_ABI_EXPORT SystemApi
{
public:
    SystemApi(SystemApi &f);
    SystemApi(SystemApi &&f);
    virtual ~SystemApi();

    /**
     *
     * @return the major version of Software.
     */
    int getMajorVersion();

    /**
     *
     * @return the minor version of Software.
     */
    int getMinorVersion();

    /**
     *
     * @return the bug fix version of Software.
     */
    int getBugfixVersion();

    /**
     *
     * @return the build number of Software.
     */
    int getBuildNumber();

    /**
     * Returns the {@link Locale} for the system.
     *
     * @return Locale.
     */
    std::string getLocale();

    /**
     * Returns the {@link Locale} for the programming language used in the
     * system.
     *
     * @return Locale.
     */
    std::string getLocaleForProgrammingLanguage();

    /**
     * Returns the UnitType of the system as an enum.
     *
     * @return UnitType.
     */
    UnitType getUnitType();

    /**
     *
     * @return the serial number of the robot
     */
    std::string getSerialNumber();

    /**
     * @return <code>true</code> if controlling the real, physical robot and
     * <code>false</code> if in simulation mode.
     */
    bool isRealRobot();

    /**
     * @return Max Positions of robot joints in radian
     */
    std::vector<double> getJointMaxPositions();
    std::vector<double> getJointMinPositions();
    double getJointDefaultVelocity();
    double getJointDefaultAcceleration();
    std::vector<double> getJointMaxVelocity();
    std::vector<double> getJointMaxAcceleration();

    double getTcpDefaultVelocity();
    double getTcpDefaultAcceleration();
    double getTcpMaxVelocity();
    double getTcpMaxAcceleration();
    double getTcpMaxDistance();

    double getMaxPayloadMass();
    double getMaxPayloadDistance();

    std::vector<double> getHomeDefaultPosition();
    std::vector<double> getHomePackagePosition();

    std::unordered_map<std::string, std::vector<double>> getRobotDhParams(
        bool is_real);

    IoConfigure getToolDigitalIOConfigure();

    IoConfigure getStandardDigitalInputConfigure();
    IoConfigure getConfigurableDigitalInputConfigure();

    IoConfigure getStandardDigitalOutputConfigure();
    IoConfigure getConfigurableDigitalOutputConfigure();

    IoConfigure getStandardAnalogInputConfigure();
    IoConfigure getToolAnalogInputConfigure();

    IoConfigure getStandardAnalogOutputConfigure();
    IoConfigure getToolAnalogOutputConfigure();

    IoConfigure getStaticLinkInputConfigure();
    IoConfigure getStaticLinkOutputConfigure();

private:
    friend class DataSwitch;
    SystemApi();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // SYSTEMAPI_H
