#ifndef AUBO_SCOPE_TCP_CONTRIBUTION_MODEL_H
#define AUBO_SCOPE_TCP_CONTRIBUTION_MODEL_H

#include <aubo_caps/domain/tcp/tcp.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(TcpContributionModel);

/**
 * <p>
 * This interface provides functionality for adding, updating and removing TCPs
 * in AuboScope.
 * </p>
 *
 * <b>Note: </b> This interface must only be used in an installation
 * contribution/
 */
class ARCS_ABI_EXPORT TcpContributionModel
{
public:
    TcpContributionModel(TcpContributionModel &f);
    TcpContributionModel(TcpContributionModel &&f);
    virtual ~TcpContributionModel();

    /**
     * Add a TCP to the current AuboScope installation. This makes it selectable
     * by the end user. The TCP is not modifiable by the end user.
     *
     * @param idKey The key to identify this TCP by. The key is for this AuboCap
     * only, i.e. it only has to be unique for this AuboCap and not "globally"
     * for other aubo_studio plugins.
     * @param suggested_name Suggested name for the tcp/ Valid names must match
     * regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters. The final
     * name can be retrieved from the returned TCP instance.
     * @param offset The offset of the TCP with respect to the tool output
     * flange
     * @return The TCP created and registered in AuboScope.
     * @throws TCPAlreadyAddedException If called twice using the same
     * <code>idKey</code>. Use {@link #getTCP(std::string)} to check if the TCP
     * has already been added to the current installation. Use {@link
     * #updateTCP(std::string, Pose)} to update the TCP if the offset of the TCP
     * has changed.
     * @throws IllegalTCPNameException If the suggested name does not match
     * required regex.
     * @throws IllegalArgumentException If the pose specified for the offset is
     * not inside the valid range of a TCP offset as defined by AuboScope.
     */
    TCPPtr addTcp(const std::string &idKey, const std::string &suggested_name,
                  const std::vector<double> &offset);

    /**
     * Returns the TCP previously added by this AuboCap using the same
     * <code>idKey</code>. Use this to verify if the TCP is present in the
     * current installation.
     *
     * @param idKey The key to identify this TCP by.
     * @return The TCP previously added by this AuboCap.
     *         Returns <code>null</code> if no TCP exists in current
     * installation.
     */
    TCPPtr getTcp(const std::string &idKey);

    /**
     * Update the offset of a TCP already added by this AuboCap.
     *
     * @param idKey The key to identify the tcp/ A TCP must have been added
     * prior to this method call using the same key.
     * @param newOffset The new offset to set for the TCP with respect to the
     * tool output flange.
     * @throws TCPNotFoundException If no TCP exists with the provided
     * <code>idKey</code>.
     * @throws IllegalArgumentException If the pose specified for the offset is
     * not inside the valid range of a TCP offset as defined by AuboScope.
     */
    void updateTcp(const std::string &idKey,
                   const std::vector<double> &newOffset);

    /**
     * Remove a TCP added by this AuboCap from AuboScope. Program nodes using
     * the TCP will be become undefined because the TCP is no longer resolvable.
     *
     * @param idKey The key used to add the TCP with.
     * @throws TCPNotFoundException If no TCP exists with the provided
     * <code>idKey</code>.
     */
    void removeTcp(const std::string &idKey);

private:
    friend class DataSwitch;
    TcpContributionModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // TCPCONTRIBUTIONMODEL_H
