#ifndef AUBO_SCOPE_KEYBOARD_INPUT_VALIDATOR_H
#define AUBO_SCOPE_KEYBOARD_INPUT_VALIDATOR_H

#include <string>
#include <functional>
#include <float.h>

#include <QObject>

#include <aubo_caps/domain/value/expression/expression.h>

namespace arcs {
namespace aubo_scope {

/**
 * <p>
 * Interface representing the input validators created by {@link
 * InputValidationFactory}. This factory provides a set of standard input
 * validators that can be used for validating the input entered by the user
 * using the virtual keyboard/keypad.
 * </p>
 *
 * If the standard validators available in the {@link InputValidationFactory}
 * does not satisfy your needs, you are free to implement your own custom
 * validator.
 *
 * @param <T> the (generic) type parameter for the interface representing the
 * type of input data being validated (e.g. Integer or Double).
 */
struct InputValidator
{
    /**
     * @param value to be validated.
     * @return <code>true</code> if value is valid.
     */
    virtual bool isValid(std::string value) = 0;

    /**
     * Returns a meaningful message in case the value is not valid.
     *
     * @param value the invalid value. Can be included in the message if it
     * makes sense.
     * @return message.
     */
    virtual std::string getMessage(std::string value) = 0;
};

struct IntegerRangeValidator : InputValidator
{
    IntegerRangeValidator(int min_value, int max_value);

    IntegerRangeValidator(std::function<std::pair<int, int>()> range_func);

    bool isValid(std::string value) override;

    std::string getMessage(std::string value) override;

private:
    void updateValue();

private:
    int min_value_{ INT_MIN };
    int max_value_{ INT_MAX };

    std::function<std::pair<int, int>()> range_func_{ nullptr };
};

struct UIntegerRangeValidator : InputValidator
{
    UIntegerRangeValidator(uint32_t min_value, uint32_t max_value);

    UIntegerRangeValidator(
        std::function<std::pair<uint32_t, uint32_t>()> range_func);

    bool isValid(std::string value) override;

    std::string getMessage(std::string value) override;

private:
    void updateValue();

private:
    uint32_t min_value_{ 0 };
    uint32_t max_value_{ UINT_MAX };

    std::function<std::pair<uint32_t, uint32_t>()> range_func_{ nullptr };
};

struct DoubleRangeValidator : InputValidator
{
    DoubleRangeValidator(double minValue, double maxValue,
                         bool is_min_inclusive = true,
                         bool is_max_inclusive = true);

    DoubleRangeValidator(std::function<std::pair<double, double>()> range_func);

    DoubleRangeValidator(
        std::function<std::tuple<double, double, bool, bool>()> range_func);

    bool isValid(std::string value) override;

    std::string getMessage(std::string value) override;

private:
    void updateValue();

private:
    double min_value_{ DBL_MIN };
    double max_value_{ DBL_MAX };
    std::function<std::pair<double, double>()> range_func_{ nullptr };
    std::function<std::tuple<double, double, bool, bool>()>
        range_func_with_flags_{ nullptr };
    bool is_min_inclusive_{ true }; // 最小值是否闭区间
    bool is_max_inclusive_{ true }; // 最大值是否闭区间
};

struct StringLengthValidator : InputValidator
{
    StringLengthValidator(size_t min_length, size_t max_length);

    StringLengthValidator(
        std::function<std::pair<size_t, size_t>()> range_func);

    bool isValid(std::string value) override;

    std::string getMessage(std::string value) override;

private:
    void updateValue();

private:
    size_t min_length_{ 0 };
    size_t max_length_{ ULONG_MAX };
    std::function<std::pair<size_t, size_t>()> range_func_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif
