#ifndef AUBO_CAPS_INTERFACE_RENDER_INTERFACE_H
#define AUBO_CAPS_INTERFACE_RENDER_INTERFACE_H

#include <vector>
#include <aubo_caps/class_forward.h>

class QGridLayout;

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(RenderInterface);

class ARCS_ABI_EXPORT RenderInterface
{
public:
    enum ShowOptions : uint16_t
    {
        None = 0,
        Ground = 0x01,
        TargetRobot = 0x02,
        RealRobot = 0x04,
        Coordinate = 0x08,
        Track = 0x10,
        Point = 0x20,
        View = 0x40,
        Obstacle = 0x80,
        SafetyPlane = 0x0100,
        ToolPosition = 0x0200,
        PointCloud = 0x0400,
        All = 0xFFFF,
    };

public:
    RenderInterface(RenderInterface &f);
    RenderInterface(RenderInterface &&f);
    virtual ~RenderInterface();

    /**
     * 设置 Render 窗口嵌入/移除(isEmbeded) 布局 gly 中
     *
     * @param is_embeded 是否嵌入
     * @param gly 布局
     * @return 无
     */
    void setRenderEmbeded(bool is_embeded, QGridLayout *gly);

    /**
     * 设置显示模式，控制各种元素的显隐
     *
     * @param options 选项值，详情查看 ShowOptions 枚举
     * @return 无
     */
    void setShowOptions(uint16_t options);

    /**
     * 添加/修改点云
     *
     * @param name 点云名称
     * @param points 点集合
     * @param color 点云颜色 rgba
     * @return 无
     */
    void addPointCloud(const std::string &name,
                       const std::vector<std::vector<double>> &points,
                       const std::vector<double> &color);

    /**
     * 移除指定名称的点云
     *
     * @param name 点云名称
     * @return 无
     */
    void removePointCloud(const std::string &name);

    /**
     * 查找制定名称的点云是否存在
     *
     * @param name 点云名称
     * @retval true 存在
     * @retval false 不存在
     */
    bool hasPointCloud(const std::string &name);

    /**
     * 移除所有的点云
     *
     * @param 无
     * @retval 无
     */
    void clearPointCloud();

    /**
     * 添加/修改路点
     *
     * @param wp 代表路点的唯一对象
     * @param pose 路点位姿 x,y,z,rx,ry,rz
     * @param color 路点颜色 {r, g, b, a},  范围是 0.0~1.0
     * @param axis_visible 路点上是否显示位姿方向箭头
     * @retval true 成功
     * @retval false 失败
     */
    bool addWaypoint(void *wp, const std::vector<double> &pose,
                     const std::vector<double> &color = { 0.0, 1.0, 0.0, 1.0 },
                     bool axis_visible = false);

    /**
     * 移除指定路点
     *
     * @param wp 代表路点的唯一对象
     * @retval 无
     */
    void removeWaypoint(void *wp);

    /**
     * 查找指定路点是否存在
     *
     * @param wp 代表路点的唯一对象
     * @retval true 存在
     * @retval false 不存在
     */
    bool hasWaypoint(void *wp);

    /**
     * 移除所有的路点
     *
     * @param 无
     * @retval 无
     */
    void clearWaypoints();

    /**
     * 添加/修改指定路点的交融轨迹
     *
     * @param wp 代表路点的唯一对象
     * @param track 路点位姿 x,y,z,rx,ry,rz 集合
     * @retval 无
     */
    void setMoveTrack(void *wp, const std::vector<std::vector<double>> &track);
    void setBlendTrack(void *wp, const std::vector<std::vector<double>> &track);
    void setMoveTrack(void *wp, const std::vector<std::vector<double>> &track,
                      const std::vector<double> &color);
    void setBlendTrack(void *wp, const std::vector<std::vector<double>> &track,
                       const std::vector<double> &color);

    /**
     * 添加/修改坐标轴
     *
     * @param name 坐标轴名称
     * @param pose 位姿 x,y,z,rx,ry,rz
     * @retval 无
     */
    void addCoordinate(const std::string &name,
                       const std::vector<double> &pose);

    /**
     * 移除指定坐标轴
     *
     * @param name 坐标轴名称
     * @retval 无
     */
    void removeCoordinate(const std::string &name);

    /**
     * 查找指定坐标轴是否存在
     *
     * @param name 坐标轴名称
     * @retval true 存在
     * @retval false 不存在
     */
    bool hasCoordinate(const std::string &name);

    /**
     * 添加/修改线
     *
     * @param name 线名称
     * @param link_num: -1: TCP; 0: 底座; 1~6: 关节;
     * @param pose: 点的参考坐标系, 默认 6 个 0
     * @param points 点集合
     * @param color 线颜色 {r, g, b, a}, 范围是 0.0~1.0
     * @retval 无
     */
    void addLine(const std::string &name,
                 const std::vector<std::vector<double>> &points,
                 const std::vector<double> &color);
    void addLine(const std::string &name, int link_num,
                 const std::vector<double> &pose,
                 const std::vector<std::vector<double>> &points,
                 const std::vector<double> &color);

    /**
     * 移除指定线
     *
     * @param name 线名称
     * @retval 无
     */
    void removeLine(const std::string &name);

    /**
     * 移除所有的线
     *
     * @param 无
     * @retval 无
     */
    void clearLines();

    /**
     * 添加/修改平面
     *
     * @param name 平面名称
     * @param link_num: -1: TCP; 0: 底座; 1~6: 关节;
     * @param pose 位置
     * @param size 平面大小（长，宽）
     * @param color 线颜色 {r, g, b, a}, 范围是 0.0~1.0
     * @retval 无
     */
    void addPlane(const std::string &name, const std::vector<double> &pose,
                  const std::vector<double> &size,
                  const std::vector<double> &color);
    void addPlane(const std::string &name, int link_num,
                  const std::vector<double> &pose,
                  const std::vector<double> &size,
                  const std::vector<double> &color);

    /**
     * 移除指定平面
     *
     * @param name 名称
     * @retval 无
     */
    void removePlane(const std::string &name);

    /**
     * 移除所有的平面
     *
     * @param 无
     * @retval 无
     */
    void clearPlanes();

    /**
     * 添加/修改立方体
     *
     * @param name 立方体名称
     * @param link_num: -1: TCP; 0: 底座; 1~6: 关节;
     * @param pose 位置
     * @param size 立方体大小（长，宽，高）
     * @param color 立方体颜色 {r, g, b, a}, 范围是 0.0~1.0
     * @param type 立方体绘制类型  0 代表画 6 个平面，1 代表只画边框线
     * @retval 无
     */
    void addCube(const std::string &name, const std::vector<double> &pose,
                 const std::vector<double> &size,
                 const std::vector<double> &color, int type = 0);
    void addCube(const std::string &name, int link_num,
                 const std::vector<double> &pose,
                 const std::vector<double> &size,
                 const std::vector<double> &color, int type = 0);

    /**
     * 添加/修改立方体
     *
     * @param name 立方体名称
     * @param pose 位置
     * @param length_min
     * @param length_max
     * @param width_min
     * @param width_max
     * @param height_min
     * @param height_max
     * @param color 立方体颜色 {r, g, b, a}, 范围是 0.0~1.0
     * @param type 立方体绘制类型  0 代表画 6 个平面，1 代表只画边框线
     * @retval 无
     */
    void addCube(const std::string &name, const std::vector<double> &pose,
                 float length_min, float length_max, float width_min,
                 float width_max, float height_min, float height_max,
                 const std::vector<double> &color, int type = 0);
    void addCube(const std::string &name, int link_num,
                 const std::vector<double> &pose, float length_min,
                 float length_max, float width_min, float width_max,
                 float height_min, float height_max,
                 const std::vector<double> &color, int type = 0);

    /**
     * 移除指定立方体
     *
     * @param name 立方体名称
     * @retval 无
     */
    void removeCube(const std::string &name);

    /**
     * 移除所有的立方体
     *
     * @param 无
     * @retval 无
     */
    void clearCubes();

    // 模型文件
    /**
     * @brief addMesh 添加任意模型文件
     * @param name: 唯一标识
     * @param link_num: -1: TCP; 0: 底座; 1~6: 关节;
     * @param model_path: 3ds 文件绝对路径
     * @param pose: 模型文件安装位置 {x, y, z, rx, ry, rz}
     * @return 0: 成功; 其他失败
     */
    int addMesh(const std::string &name, int link_num,
                const std::string &model_path, const std::vector<double> &pose);

    /**
     * @brief getMeshes 获取关节 link_num 的外设模型名字
     * @param link_num: -1: TCP; 0: 底座; 1~6: 关节;
     * @return 名字列表
     */
    std::vector<std::string> getMeshes(int link_num);
    int removeMesh(const std::string &name);
    int clearMeshes(int link_num);

private:
    friend class DataSwitch;
    RenderInterface();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_CAPS_INTERFACE_RENDER_INTERFACE_H
