#ifndef AUBO_SCOPE_USER_INTERACTION_H
#define AUBO_SCOPE_USER_INTERACTION_H

#include <vector>
#include <functional>

#include <aubo_caps/class_forward.h>
#include <aubo_caps/domain/user_interface/keyboard_manager.h>
#include <aubo_caps/domain/user_interface/render_interface.h>

namespace arcs {
namespace aubo_scope {

ARCS_CLASS_FORWARD(UserInteraction);
ARCS_CLASS_FORWARD(TCP);

using RobotPositionCallback = std::function<void(
    bool accepted, const std::vector<double> &pose,
    const std::vector<double> &q, const std::vector<double> &offset)>;
using RobotMovementCallback = std::function<void(bool accepted, int error)>;

/**
 * This interface provides functionality for requesting input or actions from
 * end users
 */
class ARCS_ABI_EXPORT UserInteraction
{
public:
    UserInteraction(UserInteraction &f);
    UserInteraction(UserInteraction &&f);
    virtual ~UserInteraction();

    /**
     * <p>
     * Request the end user to use the robot to define a robot position.
     * Override the
     * {@link RobotPositionCallback2#onOk(PositionParameters)} method to execute
     * code once the end user is done. Optionally override the {@link
     * RobotPositionCallback2#onCancel()} method to execute code if the end user
     * cancels the operation.
     * </p>
     *
     * <p>
     * This method is asynchronous, i.e. the method will return immediately.
     * Only when the end user is done, either
     * {@link RobotPositionCallback2#onOk(PositionParameters)} or {@link
     * RobotPositionCallback2#onCancel()} will be called.
     * </p>
     *
     * NOTE: This functionality should not be used in a toolbar contribution
     * (see {@link ToolbarService}).
     *
     * @param callback the instance callbacks will be made to.
     */
    void getUserDefinedRobotPosition(RobotPositionCallback callback);

    /**
     * 指定 tcpOfffset
     *
     * @param tcpOffset
     * @param callback
     */
    void getUserDefinedRobotPosition(const std::vector<double> &tcpOffset,
                                     RobotPositionCallback callback);

    /**
     * 指定 tcpOfffset
     *
     * @param tcpOffset
     * @param callback
     */
    void getUserDefinedRobotPosition(const std::string &tcpOffset,
                                     RobotPositionCallback callback);

    /**
     * 指定 tcpOfffset
     *
     * @param tcpOffset
     * @param callback
     */
    void getUserDefinedRobotPosition(const TCPPtr tcpOffset,
                                     RobotPositionCallback callback);

    /**
     * <p>
     * This method provides a factory for creating keyboard inputs which are
     * used to configure a virtual keyboard/keypad and to request it to be
     * displayed for a Swing GUI component.
     * </p>
     *
     * NOTE: This functionality is only relevant for AuboCap nodes with a
     * Swing-based user interface (see {@link ProgramNodeService},
     * {@link InstallationNodeService} and {@link ToolbarService}).
     *
     * @return factory providing access to keyboard inputs.
     */
    KeyboardManagerPtr getKeyboardManager();

    /**
     * <p>
     * Transition to the move tab widget to allow/request the end user to move
     * the robot (either automatically or manually) to a desired target position
     * specified by joint positions.
     * </p>
     *
     * <p>
     * The Automove function will move the robot to the target joint positions
     * linearly in joint space.
     * </p>
     *
     * <b>NOTE:</b> This method does not take the currently active TCP offset
     * into account, i.e. the Automove function will have the specified joint
     * positions as target.
     *
     * @param q The target joint positions to move to.
     * @param callback The instance which callbacks will be made to.
     * @throws IllegalStateException If the plugin screen is not active
     * (visible).
     */
    int requestUserToMoveJoint(const std::vector<double> &q,
                               RobotMovementCallback callback);
    int requestUserToMoveJoint(const std::string &q,
                               RobotMovementCallback callback);
    /**
     * @param pose The target pose the robot TCP will move to.
     * @param q The reference joint Angle.
     * @param callback The instance which callbacks will be made to.
     * @throws IllegalStateException If the plugin screen is not active
     * (visible).
     */
    int requestUserToMoveJoint(const std::vector<double> &pose,
                               const std::vector<double> &q,
                               RobotMovementCallback callback);

    /**
     * <p>
     * Transition to the move tab widget to allow/request the end user to move
     * the robot (either automatically or manually) to a desired target position
     * specified by a pose.
     * </p>
     *
     * <p>
     * The Automove function will move the robot TCP to the target position
     * linearly in Cartesian space. If this it not possible, the robot will move
     * to the target linearly in joint space.
     * </p>
     *
     * The current joint positions of the robot will be used as starting point
     * for inverse kinematics to calculate a target joint vector corresponding
     * to desired pose, taking the currently active TCP offset into account.
     *
     * @param pose The target pose the robot TCP will move to.
     * @param callback The instance which callbacks will be made to.
     * @throws IllegalStateException If the plugin screen is not active
     * (visible).
     */
    QT_DEPRECATED int requestUserToMoveLine(const std::vector<double> &pose,
                                            RobotMovementCallback callback);

    /**
     * <p>
     * Transition to the move tab widget to allow/request the end user to move
     * the robot (either automatically or manually) to a desired target position
     * specified by a pose.
     * </p>
     *
     * <p>
     * The Automove function will move the robot TCP to the target position
     * linearly in Cartesian space. If this it not possible, the robot will move
     * to the target linearly in joint space.
     * </p>
     *
     * The current joint positions of the robot will be used as starting point
     * for inverse kinematics to calculate a target joint vector corresponding
     * to desired pose, taking the currently active TCP offset into account.
     *
     * @param pose The target pose the robot TCP will move to.
     * @param tcpOffset
     * @param callback The instance which callbacks will be made to.
     * @throws IllegalStateException If the plugin screen is not active
     * (visible).
     */
    int requestUserToMoveLine(const std::vector<double> &pose,
                              const std::vector<double> &tcpOffset,
                              RobotMovementCallback callback);

    /**
     * 获取机器人缩略图
     */
    QPixmap *getRobotThumb(const std::vector<double> &q,
                           const std::vector<double> &pose);

    /// 界面按钮是否启用操作记录
    void setOperationLogDisplayName(QWidget *pbn, const std::string &name);

    /**
     * @return This method provides robot model drawing objects.
     */
    RenderInterfacePtr getRenderInterface();

private:
    friend class DataSwitch;
    UserInteraction();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // USERINTERACTION_H
