﻿#ifndef AUBO_SCOPE_EXPRESSION_H
#define AUBO_SCOPE_EXPRESSION_H

#include <string>
#include <vector>

#include <aubo_caps/domain/variable/variable.h>
#include <aubo_caps/domain/feature/feature.h>
#include <aubo_caps/domain/waypoint/waypoint.h>
#include <aubo_caps/domain/io/io.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(Expression);

/**
 * <p>
 * An expression can be used for, e.g. configuration of various built-in program
 * nodes, such as If and Wait nodes. Expressions are typically conditional
 * expressions (which evaluates to true or false).
 * </p>
 *
 * An Expression is built using the {@link ExpressionBuilder} interface.
 */
class ARCS_ABI_EXPORT Expression
{
public:
    Expression(Expression &f);
    Expression(Expression &&f);
    virtual ~Expression();

    std::string toString();

    /**
     * Append a string part of the expression. This can be anything that is
     * valid in an expression or even the full expression if it does not contain
     * references to any entities (such as variables and features among others).
     *
     * @param expressionPart the expression part to be appended.
     * @return the ExpressionBuilder
     */
    void append(const std::string &expressionPart);

    /**
     * Append a variable object to the expression.
     *
     * @param variable the variable to append.
     * @return the ExpressionBuilder
     */
    void appendVariable(VariablePtr variable);

    /**
     * Append a Feature object to the expression. When this part is evaluated
     * the pose of the Feature is used.
     *
     * @param feature the Feature to append
     * @return the ExpressionBuilder
     */
    void appendFeature(FeaturePtr feature);

    /**
     * Append a waypoint to the expression. When this part is evaluated the pose
     * of the waypoint is used.
     *
     * @param waypoint the waypoint to append
     * @return the ExpressionBuilder
     */
    void appendWaypoint(WaypointPtr waypoint);

    /**
     * Append an I/O object to the expression.
     *
     * @param io the I/O to append.
     * @return the ExpressionBuilder
     */
    void appendIo(IoPtr io);

private:
    friend class DataSwitch;
    Expression();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_EXPRESSION_H
